<?php

declare(strict_types=1);

namespace Drupal\commerce_zero_out_tax\OrderProcessor;

use Drupal\commerce_order\Adjustment;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\Entity\OrderItemInterface;
use Drupal\commerce_order\OrderProcessorInterface;
use Drupal\commerce_price\Price;

/**
 * Removes non-included tax adjustments when the zero-out-tax flag is set.
 */
class ZeroOutTaxOrderProcessor implements OrderProcessorInterface {

  /**
   * {@inheritdoc}
   */
  public function process(OrderInterface $order): void {
    $config = $order->getData('zero_out_tax');

    // Only act when the promotion offer ran this refresh.
    if (!$config) {
      return;
    }

    $this->removeTaxAdjustments($order, $config);

    foreach ($order->getItems() as $orderItem) {
      $this->removeTaxAdjustments($orderItem, $config);
    }
  }

  /**
   * Removes tax adjustments from the given order or order item.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface|\Drupal\commerce_order\Entity\OrderItemInterface $object
   *   The order or order item.
   * @param array $config
   *   The configuration array from the promotion offer, containing the modes
   *   of tax adjustments to remove.
   */
  protected function removeTaxAdjustments(OrderInterface|OrderItemInterface $object, array $config): void {
    $to_remove = [];
    $zero_adjustments = [];
    foreach ($object->getAdjustments(['tax']) as $tax_adjustment) {
      $is_included = $tax_adjustment->isIncluded();
      if ($is_included && !empty($config['included']) || !$is_included && !empty($config['non_included'])) {
        $to_remove[] = $tax_adjustment;
        $zero_adjustments[] = new Adjustment([
          'type' => 'tax',
          'label' => $tax_adjustment->getLabel() . ' ' . t('(removed tax)'),
          'amount' => new Price('0', $tax_adjustment->getAmount()->getCurrencyCode()),
          'included' => $tax_adjustment->isIncluded(),
          'source_id' => $tax_adjustment->getSourceId(),
        ]);
      }
    }

    foreach ($to_remove as $tax_adjustment) {
      $object->removeAdjustment($tax_adjustment);
    }

    foreach ($zero_adjustments as $zero_adjustment) {
      $object->addAdjustment($zero_adjustment);
    }
  }

}
