<?php

declare(strict_types=1);

namespace Drupal\commerce_zero_out_tax\Plugin\Commerce\PromotionOffer;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_promotion\Entity\PromotionInterface;
use Drupal\commerce_promotion\Plugin\Commerce\PromotionOffer\PromotionOfferBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Promotion offer that marks the order to have tax zeroed out.
 *
 * @CommercePromotionOffer(
 *   id = "zero_out_tax_offer",
 *   label = @Translation("Zero out tax"),
 *   entity_type = "commerce_order"
 * )
 */
class ZeroOutTaxOffer extends PromotionOfferBase {

  /**
   * {@inheritdoc}
   */
  public function apply(EntityInterface $entity, PromotionInterface $promotion): void {
    if (!$entity instanceof OrderInterface) {
      return;
    }

    if (empty($this->configuration['modes'])) {
      return;
    }

    // The pre-processor will clear it each refresh.
    $entity->setData('zero_out_tax', array_fill_keys($this->configuration['modes'], TRUE));
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    // By default, strip only non-included taxes (common US behavior).
    return ['modes' => ['non_included']] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['modes'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Tax adjustments to remove'),
      '#options' => [
        'non_included' => $this->t('Non-included taxes (added on top)'),
        'included' => $this->t('Included taxes (built into prices)'),
      ],
      '#default_value' => $this->configuration['modes'],
      '#required' => TRUE,
      '#description' => $this->t('Choose which tax adjustments to remove when this promotion applies.'),
      'non_included' => [
        '#description' => $this->t('Removes sales taxes that are added on top of prices (e.g., typical US sales tax).'),
      ],
      'included' => [
        '#description' => $this->t('Removes tax adjustments that are included in unit prices (e.g., VAT/GST included pricing).'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);
    $selected = array_keys(array_filter((array) $form_state->getValue('modes')));
    $this->configuration['modes'] = $selected;
  }

}
