<?php

namespace Drupal\commercetools_content\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\commercetools\CommercetoolsApiServiceInterface;
use Drupal\commercetools\CommercetoolsLocalization;
use Drupal\commercetools\CommercetoolsProducts;
use Drupal\commercetools\CommercetoolsService;
use Drupal\commercetools_content\CommercetoolsAjaxInterface;
use Drupal\commercetools_content\Form\ContentSettingsForm;
use Drupal\commercetools_content\ProductListConfigurationDto;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a controller for the Commercetools catalog pages.
 */
class CommercetoolsCatalogController extends ControllerBase implements CommercetoolsAjaxInterface {

  /**
   * The Commercetools service.
   *
   * @var \Drupal\commercetools\CommercetoolsService
   */
  protected CommercetoolsService $ct;

  /**
   * The commercetools products service.
   *
   * @var \Drupal\commercetools\CommercetoolsProducts
   */
  protected CommercetoolsProducts $ctProducts;

  /**
   * The pager manager service.
   *
   * @var \Drupal\Core\Pager\PagerManagerInterface
   */
  protected $pagerManager;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;


  /**
   * The commercetools content component service.
   *
   * @var \Drupal\commercetools_content\Service\CommercetoolsContentComponents
   */
  protected $contentComponents;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->ct = $container->get('commercetools');
    $instance->ctProducts = $container->get('commercetools.products');
    $instance->pagerManager = $container->get('pager.manager');
    $instance->formBuilder = $container->get('form_builder');
    $instance->messenger = $container->get('messenger');
    $instance->contentComponents = $container->get('commercetools_content.content_components');
    return $instance;
  }

  /**
   * Get a catalog content according to the filters.
   *
   * @return array
   *   A render array.
   */
  public function getContent(): array {
    $output = [];
    $unavailable_data_text = $this->config(CommercetoolsService::CONFIGURATION_NAME)->get(CommercetoolsService::CONFIG_UNAVAILABLE_DATA_TEXT);
    $itemsPerPage = $this->config(CommercetoolsService::CONFIGURATION_NAME)->get(CommercetoolsService::CONFIG_ITEMS_PER_PAGE);
    if (empty($itemsPerPage) || $itemsPerPage <= 0) {
      $this->messenger->addError($this->t('Invalid configuration for items per page.'));
      return $output;
    }
    $configurationDto = new ProductListConfigurationDto(
      itemsPerPage: $itemsPerPage,
      unavailableDataText: $unavailable_data_text,
    );
    $output = $this->contentComponents->getProductListComponent($configurationDto);
    $output['#attached']['library'][] = 'commercetools_content/ajaxify_blocks';
    $output['#cache']['contexts'][] = 'url.query_args';
    return $output;
  }

  /**
   * Displays a page with the list of Commercetools products.
   */
  public function view(): array {
    return [
      '#type' => 'container',
      '#attributes' => [
        'class' => ['commercetools-product-catalog'],
      ],
      '#cache' => [
        'contexts' => ['url.query_args'],
        'tags' => [
          'config:' . CommercetoolsService::CONFIGURATION_NAME,
          'config:' . CommercetoolsLocalization::CONFIGURATION_NAME,
          'config:' . CommercetoolsApiServiceInterface::CONFIGURATION_NAME,
          'config:' . ContentSettingsForm::CONFIGURATION_NAME,
        ],
      ],
      'content' => $this->getContent(),
    ];
  }

}
