<?php

declare(strict_types=1);

namespace Drupal\commercetools_content\Plugin\Block;

use Drupal\Core\Cache\Cache;
use Drupal\commercetools\CommercetoolsService;
use Drupal\commercetools\Plugin\Block\CommercetoolsProductListBlockBase;
use Drupal\commercetools_content\Form\ContentSettingsForm;
use Drupal\commercetools_content\ProductListConfigurationDto;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a product list block.
 *
 * @Block(
 *   id = "commercetools_content_product_list",
 *   admin_label = @Translation("Product List"),
 * )
 */
class CommercetoolsContentProductListBlock extends CommercetoolsProductListBlockBase {

  /**
   * The commercetools content component service.
   *
   * @var \Drupal\commercetools_content\Service\CommercetoolsContentComponents
   */
  protected $contentComponents;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->contentComponents = $container->get('commercetools_content.content_components');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildSafe(): array {
    $unavailableDataText = $this->configFactory
      ->get(CommercetoolsService::CONFIGURATION_NAME)
      ->get(CommercetoolsService::CONFIG_UNAVAILABLE_DATA_TEXT);
    $itemsPerPage = isset($this->configuration['items_per_page']) && !empty($this->configuration['items_per_page'])
      ? (int) $this->configuration['items_per_page']
      : (int) $this->configFactory
        ->get(CommercetoolsService::CONFIGURATION_NAME)
        ->get(CommercetoolsService::CONFIG_ITEMS_PER_PAGE);
    $configurationDto = new ProductListConfigurationDto(
      style: $this->configuration['style'] ?? 'card',
      itemsPerPage: (int) $itemsPerPage,
      unavailableDataText: $unavailableDataText,
      limit: (int) $this->configuration['total_limit'],
      sortBy: $this->configuration['sort_by'] ?? 'createdAt',
      sortOrder: $this->configuration['sort_order'] ?? 'asc',
      categories: $this->configuration['categories'] ?? [],
      skus: $this->configuration['display_by_sku'] ?? [],
      customFilters: $this->configuration['custom_filters'] ?? [],
      columnsNumber: (int) $this->configuration['columns_number'],
    );

    return $this->contentComponents->getProductListComponent($configurationDto, (int) $this->configuration['product_list_index']);
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(): array {
    $cacheTags = parent::getCacheTags();
    return Cache::mergeTags($cacheTags, [
      'config:' . ContentSettingsForm::CONFIGURATION_NAME,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts(): array {
    $cacheContexts = parent::getCacheContexts();
    return Cache::mergeContexts($cacheContexts, [
      'url.query_args',
    ]);
  }

}
