class CategoriesBase extends HTMLElement {
  async connectedCallback() {
    this.loading = true;
    this.categories = [];
    this.categoriesTree = [];
    this.setPreloadMockData();
    this.renderComponent();
    this.handleFilterChange = this.handleFilterChange.bind(this);
    document.addEventListener('ct.filters.change', this.handleFilterChange);
    try {
      window.commercetools.getCategories().then((result) => {
        if (result.total > 0) {
          this.categories = result.categories;
          this.buildItemsTree();
        }
        this.renderComponent();
      });
    } catch (error) {
      new Drupal.Message().add(Drupal.t('Failed to load categories.'), {
        type: 'error',
      });
      throw error;
    }
  }

  setPreloadMockData() {
    this.categoriesTree = [
      {
        id: '1',
        name: '.',
        children: [],
      },
      {
        id: '2',
        name: '.',
        children: [],
      },
      {
        id: '3',
        name: '.',
        children: [],
      },
    ];
  }

  buildItemsTree() {
    const categories = JSON.parse(JSON.stringify(this.categories));
    this.categoriesTree = window.commercetools.buildCategoriesTree(
      categories,
      this.getActiveCategoryId(),
    );
    this.loading = false;
  }

  handleFilterChange() {
    this.buildItemsTree();
    this.renderComponent();
  }

  getActiveCategoryId() {
    return window.commercetools.getQueryParamByIndex(
      'category',
      this.componentConfig.productListIndex,
    );
  }

  findCategoryLevel(items, id, currentDepth = 1) {
    // We have to use `for` here because need the early exit.
    // eslint-disable-next-line no-restricted-syntax
    for (const node of items) {
      if (node.id === id) {
        return currentDepth;
      }
      if (Array.isArray(node.children) && node.children.length) {
        const childDepth = this.findCategoryLevel(
          node.children,
          id,
          currentDepth + 1,
        );
        if (childDepth !== null) {
          return childDepth;
        }
      }
    }
    return null;
  }

  getCategorySubtree(items, id, maxDepth = 1) {
    const prune = (nodes, d) => {
      if (d <= 0) {
        return [];
      }
      return nodes.map((node) => ({
        ...node,
        children: prune(node.children || [], d - 1),
      }));
    };

    // eslint-disable-next-line no-restricted-syntax
    for (const node of items) {
      if (node.id === id) {
        return {
          ...node,
          children: prune(node.children || [], maxDepth),
        };
      }
      if (Array.isArray(node.children) && node.children.length) {
        const subtree = this.getCategorySubtree(node.children, id, maxDepth);
        if (subtree) {
          return subtree;
        }
      }
    }

    return null;
  }

  // This method does not reference `this`, but is inherited classes use this method.
  // eslint-disable-next-line class-methods-use-this
  renderComponent() {
    throw new Error("Method 'renderComponent()' must be implemented.");
  }
}

window.commercetools.CategoriesBase = CategoriesBase;
