class Orders extends HTMLElement {
  constructor() {
    super();
    this.currentPage = 0;
    this.initialLoad = true;
    this.itemsPerPage =
      window.drupalSettings.commercetoolsDecoupled.itemsPerPage;
    this.handlePageChange = this.handlePageChange.bind(this);
  }

  async connectedCallback() {
    if (!this.itemsPerPage || this.itemsPerPage <= 0) {
      new Drupal.Message().add(
        Drupal.t('Invalid configuration for items per page.'),
        { type: 'error' },
      );
      return;
    }

    // Placeholder for the orders data.
    this.ordersData = {
      total: -1,
      orders: Array(this.itemsPerPage).fill(null),
    };

    // Loading the real orders data.
    const url = new URL(window.location);
    this.currentPage = parseInt(url.searchParams.get('page'), 10) || 0;
    await this.updateOrderData();
  }

  renderComponent() {
    this.innerHTML = /* html */ `
      <div class="container">
        <div class="row">
          <div class="col-md-12 ps-0">
            <span class="ct-order-list"></span>
          </div>
        </div>
        <div class="ct-pager"></div>
      </div>
    `;

    const orderList = document.createElement('ct-order-list');
    orderList.orders = this.ordersData.orders;
    orderList.isLoading = this.isLoading;
    this.querySelector('.ct-order-list').appendChild(orderList);

    if (this.ordersData.total >= 1) {
      const pager = document.createElement('ct-pager');
      const totalPages = Math.ceil(this.ordersData.total / this.itemsPerPage);
      if (totalPages > 1) {
        pager.setPager(this.currentPage, totalPages);
        pager.addEventListener('page-change', this.handlePageChange.bind(this));
        this.querySelector('.ct-pager').appendChild(pager);
      }
    }
    this.initialLoad = false;
  }

  async handlePageChange(event) {
    this.currentPage = event.detail.page;
    const urlParams = new URLSearchParams(window.location.search);
    urlParams.set('page', this.currentPage);
    try {
      await this.updateOrderData(urlParams);
    } catch (error) {
      new Drupal.Message().add(
        Drupal.t('An error occurred while changing the page.'),
        {
          type: 'error',
        },
      );
      throw error;
    }
  }

  async updateOrderData(newSearchParams) {
    this.isLoading = true;
    this.renderComponent();

    try {
      const response = await window.commercetools.getOrders({
        where: 'customerId="[current_customer:id]"',
        offset: this.currentPage * this.itemsPerPage,
        limit: this.itemsPerPage,
      });
      const totalPages = Math.ceil(response.total / this.itemsPerPage);
      // Check if the current page exceeds the total pages
      if (this.currentPage >= totalPages && totalPages > 0) {
        const message = Drupal.t('The requested page does not exist.');
        new Drupal.Message().add(message, { type: 'error' });
        this.currentPage = 0;
        const urlParams = new URLSearchParams(window.location.search);
        urlParams.set('page', 0);
        await this.updateProductData(urlParams);
        return;
      }
      this.ordersData = response;
      this.isLoading = false;
      this.renderComponent();
      if (newSearchParams) {
        const url = Drupal.url(
          `${drupalSettings.path.currentPath}?${newSearchParams.toString()}`,
        );
        window.history.pushState({ path: url }, '', url);
      }
    } finally {
      this.isLoading = false;
      this.renderComponent();
    }
  }
}

customElements.define('ct-orders', Orders);
