<?php

namespace Drupal\commercetools_decoupled\Controller;

use Drupal\commercetools\CommercetoolsApiServiceInterface;
use Drupal\commercetools\Exception\CommercetoolsGraphqlErrorException;
use Drupal\commercetools\Exception\CommercetoolsInvalidGraphqlRequestException;
use Drupal\Core\Cache\CacheableJsonResponse;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Path\CurrentPathStack;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a proxy endpoint to CommerceTools API.
 */
class CommercetoolsApiController extends ControllerBase {

  /**
   * CommercetoolsApiController constructor.
   *
   * @param \Drupal\commercetools\CommercetoolsApiServiceInterface $ctApi
   *   The CommerceTools API service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Path\CurrentPathStack $currentPathStack
   *   The current path stack.
   */
  public function __construct(
    protected CommercetoolsApiServiceInterface $ctApi,
    protected RequestStack $requestStack,
    protected CurrentPathStack $currentPathStack,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('commercetools.api'),
      $container->get('request_stack'),
      $container->get('path.current'),
    );
  }

  /**
   * Proxies a GraphQL response to the CommerceTools GraphQL API.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response: either a `CacheableJsonResponse` with operation data and
   *   cache dependencies, or a regular `JsonResponse` containing errors.
   *
   * @throws \Exception
   *   Thrown when the path is not an API path.
   */
  public function apiProxyGraphql(string $projectId) {
    $request = $this->requestStack->getCurrentRequest();
    if ($request->getMethod() !== 'POST') {
      throw new CommercetoolsInvalidGraphqlRequestException('Invalid method called for the GraphQL operation: ' . $request->getMethod() . '. GraphQL endpoint works only with POST method.');
    }
    $operation = json_decode($request->getContent(), associative: TRUE);
    if (!isset($operation['query'])) {
      throw new CommercetoolsInvalidGraphqlRequestException();
    }
    $operation['variables']['proxy'] = TRUE;
    try {
      $result = $this->ctApi->executeGraphQlOperation($operation['query'], $operation['variables']);
    }
    catch (CommercetoolsGraphqlErrorException $e) {
      $responseData = [
        'errors' => $e->getMessages(),
      ];
      return new JsonResponse($responseData);
    }

    $response = new CacheableJsonResponse(['data' => $result->getData()]);
    $response->addCacheableDependency($result->getCacheableMetadata());
    return $response;
  }

}
