<?php

namespace Drupal\commercetools;

use Drupal\commercetools\Cache\CacheableCommercetoolsGraphQlResponse;
use Drupal\commercetools\Event\CommercetoolsMessageEvent;
use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * The Commercetools Message processor event subscriber.
 */
class CommercetoolsMessageProcessor implements EventSubscriberInterface {

  const PROCESSOR_PER_TYPE = [
    /* Product messages */
    'ProductCreated' => 'processProduct',
    'ProductPublished' => 'processProduct',
    'ProductUnpublished' => 'processProduct',
    'ProductStateTransition' => 'processProduct',
    'ProductSlugChanged' => 'processProduct',
    'ProductImageAdded' => 'processProduct',
    'ProductRevertedStagedChanges' => 'processProduct',
    'ProductPriceKeySet' => 'processProduct',
    'ProductPriceDiscountsSet' => 'processProduct',
    'ProductPriceDiscountsSetUpdatedPrice' => 'processProduct',
    'ProductPriceExternalDiscountSet' => 'processProduct',
    'ProductPriceAdded' => 'processProduct',
    'ProductPriceChanged' => 'processProduct',
    'ProductPricesSet' => 'processProduct',
    'ProductPriceRemoved' => 'processProduct',
    'ProductPriceModeSet' => 'processProduct',
    'ProductVariantAdded' => 'processProduct',
    'ProductVariantDeleted' => 'processProduct',
    'ProductAddedToCategory' => 'processProduct',
    'ProductRemovedFromCategory' => 'processProduct',
    'ProductDeleted' => 'processProduct',
    'ProductPriceCustomFieldAdded' => 'processProduct',
    'ProductPriceCustomFieldChanged' => 'processProduct',
    'ProductPriceCustomFieldRemoved' => 'processProduct',
    'ProductPriceCustomFieldsSet' => 'processProduct',
    'ProductPriceCustomFieldsRemoved' => 'processProduct',
  ];

  /**
   * Constructs a new MessageProcessor object.
   *
   * @param \Drupal\Core\Cache\CacheTagsInvalidatorInterface $cacheTagsInvalidator
   *   The cache tags invalidator.
   */
  public function __construct(
    protected CacheTagsInvalidatorInterface $cacheTagsInvalidator,
  ) {
  }

  /**
   * Process the message.
   *
   * @param \Drupal\commercetools\Event\CommercetoolsMessageEvent $event
   *   The message event.
   */
  public function processMessage(CommercetoolsMessageEvent $event) {
    $message = $event->getMessage();
    if ($processor = self::PROCESSOR_PER_TYPE[$message['type']] ?? NULL) {
      $this->$processor($message);
    }
  }

  /**
   * Process the product related messages.
   *
   * @param array $message
   *   The message.
   */
  protected function processProduct($message) {
    $tags = [
      CacheableCommercetoolsGraphQlResponse::CACHE_TAG_PRODUCT_LIST,
    ];
    if ($productId = $message['resourceRef']['id']) {
      $tags[] = CacheableCommercetoolsGraphQlResponse::CACHE_TAG_PRODUCT_PREFIX . $productId;
    }
    $this->cacheTagsInvalidator->invalidateTags($tags);
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events[CommercetoolsMessageEvent::class][] = ['processMessage'];
    return $events;
  }

}
