<?php

declare(strict_types=1);

namespace Drupal\commercetools\Plugin\Block;

use Drupal\commercetools\CommercetoolsConfiguration;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a base class for commercetools component blocks.
 */
abstract class CommercetoolsCatalogBlockBase extends CommercetoolsBlockBase {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    $instance = parent::create(...func_get_args());
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);
    if (in_array('product_list_index', $this->getBlockConfigKeys())) {
      $form['product_list_index'] = [
        '#type' => 'number',
        '#title' => $this->t('Product List Index'),
        '#default_value' => $this->configuration['product_list_index'] ?? 0,
        '#min' => 0,
        '#description' => $this->t('A unique index number of the product list on the pages. Allows configure multiple instances of the block on the page with pager and filters work independently. Use <code>1</code> for the first block on the page, <code>2</code> for the second group of blocks, etc. Use <code>0</code> to connect the block with the main page content.'),
      ];
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    foreach ($this->getBlockConfigKeys() as $field) {
      $this->configuration[$field] = $form_state->getValue($field);
    }
  }

  /**
   * Lists managed field keys.
   *
   * @return array
   *   A list of managed field keys.
   */
  public function getBlockConfigKeys(): array {
    return [
      'product_list_index',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts(): array {
    $cacheContexts = parent::getCacheContexts();
    return Cache::mergeContexts($cacheContexts, [
      'languages:language_content',
      'url.query_args',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(): array {
    $cacheTags = parent::getCacheTags();
    return Cache::mergeTags($cacheTags, [
      'config:' . CommercetoolsConfiguration::CONFIGURATION_API,
      'config:' . CommercetoolsConfiguration::CONFIGURATION_SETTINGS,
    ]);
  }

  /**
   * Builds hierarchical category options.
   */
  protected function buildHierarchicalOptions(array $categories, string $prefix = ''): array {
    return array_reduce($categories, function ($options, $category) use ($prefix) {
      $options[$category['id']] = $prefix . $category['name'];
      if (!empty($category['children'])) {
        $options += $this->buildHierarchicalOptions($category['children'], $prefix . '- ');
      }
      return $options;
    }, []);
  }

}
