#!/usr/bin/env php
<?php

/**
 * @file
 * Regenerate the assets for the commercetools module tests.
 */

declare(strict_types=1);

// The variables will be set by the script that includes this file.
$moduleDirectory = '';
$assetsDirectory = '';

require_once 'utils.inc.php';

if (empty(getenv('CT_CLIENT_ID')) || empty(getenv('CT_CLIENT_SECRET'))) {
  throw new Exception("Please fill the CT_CLIENT_ID and CT_CLIENT_SECRET environment variables with commercetools access credentials.");
}

chdir(find_drupal_root() . '/core');

$logFile = tempnam(sys_get_temp_dir(), 'test_helpers_http_client_log_');

putenv('CT_API_MOCK_MODE=append');
putenv('CT_LOG_STORED_RESPONSES_FILE=' . $logFile);

$testCommandsStack = [
  'phpunit ' . escapeshellarg($moduleDirectory),
  'yarn test:nightwatch --tag commercetools',
];

foreach ($testCommandsStack as $command) {
  $result = passthru($command, $exitCode);
  if ($result === FALSE) {
    throw new Error("Failed to run the test command:\n$command\nExiting.");
  }
  if ($exitCode != 0) {
    throw new Error("Failed to run the test command:\n$command\nReturning exit code: $exitCode\nExiting.");
  }
}
$usedHashes = get_used_hashes_from_test_helpers_log($logFile);
$storedAssets = parse_stored_assets_files($assetsDirectory);
$storedHashes = $storedAssets['hashes'];

echo "List of all stored assets hashes:\n" . implode("\n", $storedHashes) . "\n\n";
echo "List of used hashes in the running tests:\n" . implode("\n", $usedHashes) . "\n\n";
$obsoleteFiles = array_diff($storedHashes, $usedHashes);

if ($obsoleteFiles) {
  echo "The current all tests run doesn't use these stored response assets (" . count($obsoleteFiles) . " items):\n"
    . implode("\n", $obsoleteFiles) . "\n\n";
  foreach ($obsoleteFiles as $hash) {
    $file = $assetsDirectory . '/' . $hash . '.txt';
    unlink($file);
    $fileMetadata = $assetsDirectory . '/' . $hash . '.metadata.yml';
    unlink($fileMetadata);
  }
  echo "The obsolete stored responses files and metadata deleted.\n";
}
else {
  echo "The current tests run used all stored response assets, nothing to cleanup.\n";
}

if ($storedAssets['otherFiles']) {
  echo "The current assets directory contain redundant files:\n"
    . implode("\n", $storedAssets['otherFiles']) . "\n\n";
  foreach ($storedAssets['otherFiles'] as $fileName) {
    $file = $assetsDirectory . '/' . $fileName;
    unlink($file);
  }
  echo "\nRedundant files deleted.\n";
}
