# Contributing to Drupal commercetools module

We welcome and appreciate contributions from developers like you! Your efforts
help improve the Drupal commercetools module and make it even more powerful for
he community. Whether you're fixing bugs, adding new features, or improving
documentation, your contributions are highly valued.

Thank you for taking the time to collaborate and make this project better!

The module is covered by unit and functional tests to simplify regression
testing, using PHPUnit and Nightwatch.

So you need to run tests locally to check that your changes don't break
anything, and adapt tests if you change anything, that already covered by the
current tests.

To run tests locally you need install the Selenium framework. We recommend using
[DDEV](https://ddev.com/) for development and
[ddev-selenium-standalone-chrome](https://github.com/ddev/ddev-selenium-standalone-chrome)
add-on that simplifies the local setup, but you can use any other Drupal
compatible environment for developing.

## Quick setup of the local environment using DDEV

```
# Create a new DDEV project
ddev config --docroot=web --project-type=drupal11
# Add support for running Nightwatch tests using Selenium
ddev get ddev/ddev-selenium-standalone-chrome
ddev composer create drupal/recommended-project --no-install
ddev composer install
ddev composer require drush/drush
ddev composer require drupal/core-dev --dev --update-with-all-dependencies
ddev drush site-install

# Install the module
ddev composer require drupal/commercetools
ddev drush pm:install commercetools
```

## commercetools Content submodule

The module renders the commercetools contend on the backend using Drupal API and
deliver it to the frontend as Drupal html pages.


## commercetools Decoupled submodule

The module provides decoupled blocks for the commercetools, which load and
render the content on the browser side.

## commercetools API documentation

The module uses the commercetools SDK API, mostly the GraphQL API.

Documentation references:

- [PHP SDK](https://docs.commercetools.com/sdk/php-sdk)

- [JavaScript SDK](https://docs.commercetools.com/sdk/typescript-sdk)

- [REST API
  documentation](https://docs.commercetools.com/learning-composable-commerce-developer-essentials/api-queries-query-predicates/overview)

- [GraphQL
  documentation](https://docs.commercetools.com/learning-composable-commerce-developer-essentials/graphql/overview)

- [Constructing the "where" query
  parameter](https://docs.commercetools.com/api/predicates/query)


## Running tests

The module uses commercetools API, so for almost each test we need correct
responses from the commercetools API. And for that you need the working
commercetools credentials! But the commercetools API can be rate limited or
slow, so using the remote API in tests is not a good idea.

To workaround this, the tests store the real responses from the API in files and
mock them when running tests. The responses content is stored in the directory
`tests/modules/commercetools_test/assets`.

If you write new tests or modify existing ones that modifies the commercetools
API requests (for example, add new fields to the GraphQL query), you should use
the real credentials to update the asset files. To make this work, use the
environment values to control the tests running mode:

- `CT_API_MOCK_MODE=append` - this variable enables making real requests to
  commercetools and storing responses to files.

We use one credentials from one of the demo projects for tests, so you can get
the working credentials from the file:
`modules/commercetools_demo/fixtures/demo_accounts/b2c_lifestyle.yml`

With these variables, the tests will do real requests to commercetools and store
them into assets, to make the tests run well next time without any credentials.

To run Nightwatch tests locally you need to install additional dependencies from
the `require-dev` section of the module's `composer.json` and apply patches,
also install NPM dependencies from the file `package.json`.


### Mocking commercetools API modes

You can control the mode of the commercetools API mocking by setting the values
for the env variable `CT_API_MOCK_MODE`:
- `mock` (default): reads responses only from the stored asset files, if missing
  - throws an exception.
- `append`: uses the asset files first, if missing - makes a real call and
  create/overwrite asset files.
- `store`: always makes direct calls to API and create/overwrite asset files.


### Run PHPUnit tests

```
cd web/core && phpunit ../modules/contrib/commercetools
```

### Run Nightwatch tests (End-to-end browser tests)

#### Install Nightwatch dependencies (only once)

```
cd web/core && yarn
cd web/modules/contrib/commercetools && yarn
```

#### Run Nightwatch tests

To run all Nightwatch tests - use this command:
```
cd web/core && yarn test:nightwatch --tag commercetools
```
or to run a single test - put the full path to the test like this
```
cd web/core && yarn test:nightwatch ../modules/contrib/commercetools/modules/commercetools_demo/tests/src/Nightwatch/Tests/demoContentSmokeTest.js
```

Tips:
If the project_key changed or produced a change that is not compatible with the updatedb check,
add this marker to the commit message to prevent the rollback to this commit:
'[ct-disable-updatedb-check-rollback]'

### Upgrade to Nightwatch 3.x for Drupal below 11.0.

We use Nightwatch 3.x version because 2.x has issues with finding nested
elements (`element('selector1').findElement('selector2')`).

To install Nightwatch 3 on your local machine, just open the `web/core`
directory and run:
```
yarn add --dev nightwatch@^3
```

Also, Nightwatch from version 2.5+ requires the more recent version of Selenium
container, so instead of the `seleniarm/standalone-chromium:4.1.4-20220429` -
use the image `selenium/standalone-chrome:latest` and enable W3C compatibility
by the env: `DRUPAL_TEST_WEBDRIVER_W3C=1`.

##### Run a single test on the pipeline.

To run a single specific test on the pipeline, without waiting for other tests
and steps, you can use this approach:

1. Comment the general tag for all tests in the file
`tests/src/Nightwatch/Lib/getBeforeAfterFunctions.js`:
```diff
-   let testTags = options.testTags ?? ['commercetools', moduleName];
+   // let testTags = options.testTags ?? ['commercetools', moduleName];
```

2. Override this value in the single test file that you want to run:
```diff
  ...getBeforeAfterFunctions(__filename),
+    '@tags': ['commercetools'],
```

3. Disable unnecessary steps in the pipeline by temporary extending the
`.gitlab-ci.yml` file:
```yaml
variables:
  # ...
  SKIP_COMPOSER_LINT: "1"
  SKIP_PAGES: "1"
  SKIP_PHPCS: "1"
  SKIP_PHPSTAN: "1"
  SKIP_PHPUNIT: "1"
  SKIP_STYLELINT: "1"
  SKIP_ESLINT: "1"
  SKIP_CSPELL: "1"
  # SKIP_NIGHTWATCH: "1"
  # OPT_IN_TEST_MAX_PHP: 0"
  OPT_IN_TEST_PREVIOUS_MAJOR: "0"
```

## Code Style

Follow the general code style rules from Drupal.org, and in addition to this:

- For any string variable try to use constants as much as possible, to simplify
finding the dependencies of the string and simplify changing the values.

### Hooks

Do not put any business logic into hook_* functions. Write all actual code as
static functions in classes like CommercetoolsUpdates::installCustomerIdField(),
and call them from the hook. This is required to simplify covering the code by
tests and the code re-usage.

So, keep in the hook only the data definitions, calls of functions and messages
output only. See the previous hook updates as examples.

Also, do not use constants in the `hook_update_*` functions, because the value
of the constants can be changed it time, that will break the old hook updates.

## Tips and Tricks

### Run all custom checks in the  `browser.perform(()=>{})` constructions.

All `browser.someCommand()` calls actually doesn't execute immediately, instead
they are added to the browser queue to execute later. Because of this, if you
need to perform some checks after the command is finished, this way will not work
- the assert will have the empty value:
```js
let myValue;
browser.getText('.my-selector', (result) => {
  myValue = result.value;
});
asset('Foo', myValue);
```

Instead, use this way:
```js
let myValue;
browser
  .getText('.my-selector', (result) => {
    myValue = result.value;
  });
  .perform(() => {
    asset('Foo', myValue);
  });
```
But this way will not work too:
```js
let myValue;
browser
  .getText('.my-selector', (result) => {
    myValue = result.value;
  });
  .perform(asset('Foo', myValue))
```
because the `assert()` will be executed instantly when passing the `perform()`
to the browser queue.

Also, the [async-await mode is
available](https://nightwatchjs.org/guide/writing-tests/using-es6-async.html)
but try to use the sync mode everywhere to simplify the tests flow.

### Missing stored responses

If you face an exception like this when running a test:
```
Exception: Missing the stored response file for the request with hash [md5 hash]
```
this means that you modified in the code a GraphQL request, executed during this
test run, and you need to update the mocked response for this GraphQL request.

This can be done by re-running the test with the `CT_API_MOCK_MODE=append`
environment variable, and adding to git the files with the body of the responses
from the API.

### ESLint Setup and Run
[Quick setup for custom and contrib modules](https://www.drupal.org/docs/develop/standards/javascript-coding-standards/eslint-settings#s-quick-setup-for-custom-and-contrib-modules)
