<?php

declare(strict_types=1);

namespace Drupal\commercetools_content\Plugin\Block;

use Drupal\commercetools_content\CommercetoolsAjaxTrait;
use Drupal\commercetools_content\Service\CommercetoolsAjaxHelper;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\commercetools\Plugin\Block\CommercetoolsProductFiltersBlockBase;
use Drupal\commercetools_content\Form\ContentSettingsForm;
use Drupal\commercetools_content\Service\CommercetoolsContentComponents;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a product search block.
 *
 * @Block(
 *   id = "commercetools_content_product_filters",
 *   admin_label = @Translation("Product Filters"),
 * )
 */
class CommercetoolsContentProductFiltersBlock extends CommercetoolsProductFiltersBlockBase {

  use CommercetoolsAjaxTrait;

  /**
   * The commercetools content component service.
   *
   * @var \Drupal\commercetools_content\Service\CommercetoolsContentComponents
   */
  protected CommercetoolsContentComponents $contentComponents;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): self {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->contentComponents = $container->get('commercetools_content.content_components');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return parent::defaultConfiguration() + [
      'use_ajax' => TRUE,
      CommercetoolsAjaxHelper::COMMERCETOOLS_SYSTEM_BLOCK_FORCE_UPDATE_CONFIG => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getBlockConfigKeys(): array {
    $keys = parent::getBlockConfigKeys();
    $keys[] = CommercetoolsAjaxHelper::COMMERCETOOLS_SYSTEM_BLOCK_FORCE_UPDATE_CONFIG;
    return $keys;
  }

  /**
   * {@inheritdoc}
   */
  public function buildSafe(): array {
    $targetPage = empty($this->configuration['target_page']) ? NULL : Url::fromUserInput($this->configuration['target_page']);
    $enabledFilters = $this->getEnabledFilters();
    $build = $this->contentComponents
      ->getFiltersForm(
        (int) $this->configuration['product_list_index'],
        $targetPage,
        $enabledFilters,
        array_filter($enabledFilters, function ($filter) {
          return in_array($filter['widget_type'], ['facet', 'facet_count', 'custom']);
        }),
      );

    // Add ajax class by default.
    $this->configuration['use_ajax'] = TRUE;

    return $build;
  }

  /**
   * Add option to force update system main content block.
   *
   * @param array $form
   *   Form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form State.
   *
   * @return array
   *   Form.
   */
  public function blockForm($form, FormStateInterface $form_state): array {
    $form = parent::blockForm($form, $form_state);
    $form += $this->getFormElements($this->configuration);
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(): array {
    $cacheTags = parent::getCacheTags();
    return Cache::mergeTags($cacheTags, [
      'config:' . ContentSettingsForm::CONFIGURATION_NAME,
    ]);
  }

}
