<?php

namespace Drupal\commercetools_decoupled;

use Drupal\commercetools\CommercetoolsApiServiceInterface;
use Drupal\commercetools\CommercetoolsLocalization;
use Drupal\commercetools\CommercetoolsService;
use Drupal\commercetools\Form\UiModuleSettingsFormBase;
use Drupal\commercetools\Routing\UiModulesRouteProviderBase;
use Drupal\commercetools_decoupled\Form\DecoupledSettingsForm;
use Drupal\commercetools_decoupled\Routing\RouteProvider;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;

/**
 * Commercetools Decoupled main service.
 */
class CommercetoolsDecoupled {

  /**
   * CommercetoolsApiService constructor.
   *
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\commercetools\CommercetoolsService $ct
   *   The Commercetools service.
   * @param \Drupal\commercetools\CommercetoolsApiServiceInterface $ctApi
   *   The Commercetools API service.
   * @param \Drupal\commercetools\CommercetoolsLocalization $ctLocalization
   *   The commercetools localization service.
   * @param \Drupal\commercetools_decoupled\CtAuth $ctAuth
   *   The commercetools authorization service.
   */
  public function __construct(
    protected readonly AccountInterface $currentUser,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly CommercetoolsService $ct,
    protected readonly CommercetoolsApiServiceInterface $ctApi,
    protected readonly CommercetoolsLocalization $ctLocalization,
    protected readonly CtAuth $ctAuth,
  ) {
  }

  /**
   * Returns an array with common settings required for frontend components.
   */
  public function addDecoupledSettings(array $render): array {
    $generalSettings = $this->configFactory->get(CommercetoolsService::CONFIGURATION_NAME);
    $connectionSettings = $this->configFactory->get(CommercetoolsApiServiceInterface::CONFIGURATION_NAME);
    $decoupledSettings = $this->configFactory->get(DecoupledSettingsForm::CONFIGURATION_NAME);
    $localizationSettings = $this->configFactory->get(CommercetoolsLocalization::CONFIGURATION_NAME);

    $oldSettings = $render['#attached']['drupalSettings']['commercetoolsDecoupled'] ?? [];
    $render['#attached']['drupalSettings']['commercetoolsDecoupled'] = $oldSettings + [
      'catalogPath' => $decoupledSettings->get(UiModuleSettingsFormBase::CONFIG_CATALOG_PATH),
      'cartPath' => $decoupledSettings->get(UiModuleSettingsFormBase::CONFIG_CART_PATH),
      'checkoutPath' => $decoupledSettings->get(UiModuleSettingsFormBase::CONFIG_CHECKOUT_PATH),
      'orderPathPrefix' => Url::fromRoute(RouteProvider::ROUTE_PREFIX . UiModulesRouteProviderBase::PAGE_USER_ORDERS_ROUTE, ['user' => $this->currentUser->id()])->toString(),
      'apiUrl' => $decoupledSettings->get(DecoupledSettingsForm::CONFIG_API_PROXY_PATH) ?: CommercetoolsApiServiceInterface::API_PROXY_PATH_DEFAULT,
      'projectKey' => $connectionSettings->get(CommercetoolsApiServiceInterface::CONFIG_PROJECT_KEY),
      'locale' => $localizationSettings->get(CommercetoolsLocalization::CONFIG_LANGUAGE),
      'languages' => $this->ctLocalization->getLanguageFallbacks(),
      'priceCountry' => $localizationSettings->get(CommercetoolsLocalization::CONFIG_COUNTRY),
      'priceCurrency' => $localizationSettings->get(CommercetoolsLocalization::CONFIG_CURRENCY),
      'priceGroup' => $generalSettings->get(CommercetoolsService::CONFIG_PRICE_CUSTOMER_GROUP),
      'priceChannel' => $localizationSettings->get(CommercetoolsLocalization::CONFIG_CHANNEL),
      'attributes' => $this->ct->getEnabledAttributes(),
      'itemsPerPage' => $generalSettings->get(CommercetoolsService::CONFIG_ITEMS_PER_PAGE),
      'cardImageStyle' => $generalSettings->get(CommercetoolsService::CONFIG_CARD_IMAGE_STYLE),
      'unavailableDataText' => $generalSettings->get(CommercetoolsService::CONFIG_UNAVAILABLE_DATA_TEXT),
      'apiRegion' => $this->ctApi->getApiUrl('region'),
      'resetUri' => Url::fromRoute(RouteProvider::ROUTE_PREFIX . UiModulesRouteProviderBase::PAGE_RESET_CART_ROUTE)->toString(),
    ];

    $render['#cache']['tags'] = Cache::mergeTags(
      $render['#cache']['tags'] ?? [],
      $this->getDrupalSettingsCacheTags()
    );

    $render['#cache']['contexts'] = Cache::mergeContexts($render['#cache']['contexts'] ?? [], [
      'languages:language_content',
    ]);

    $useDirectCtApiConnections = $decoupledSettings->get(DecoupledSettingsForm::CONFIG_USE_DIRECT_CT_API_CONNECTIONS);
    if ($useDirectCtApiConnections) {
      $render['#attached']['drupalSettings']['commercetoolsDecoupled']['feApiAuthorizationHeader'] = $this->ctAuth->getAuthorizationHeader();
      $render['#cache']['max-age'] = !empty($render['#cache']['max-age'])
        ? Cache::mergeMaxAges($this->ctAuth->getCacheExpire(), $render['#cache']['max-age'])
        : $this->ctAuth->getCacheExpire();
    }

    return $render;
  }

  /**
   * Returns an array with settings cache tags.
   */
  public function getDrupalSettingsCacheTags(): array {
    return [
      'config:' . CommercetoolsService::CONFIGURATION_NAME,
      'config:' . CommercetoolsApiServiceInterface::CONFIGURATION_NAME,
      'config:' . DecoupledSettingsForm::CONFIGURATION_NAME,
      'config:' . CommercetoolsLocalization::CONFIGURATION_NAME,
    ];

  }

}
