<?php

namespace Drupal\commercetools_decoupled\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Subscribes to kernel exceptions events and generates a Span Event.
 */
class CommercetoolsDecoupledExceptionSubscriber implements EventSubscriberInterface {
  use StringTranslationTrait;

  const REQUEST_FORMAT_COMMERCETOOLS_GRAPHQL = 'commercetools_graphql';

  /**
   * Constructs the CommercetoolsDecoupledExceptionSubscriber object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly LoggerInterface $logger,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
        // Set weight to 60 to execute before the ExceptionLoggingSubscriber
        // that logs the not found pages.
      KernelEvents::EXCEPTION => ['onException', 60],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function onException(ExceptionEvent $event) {
    // Act only on Commercetools related exceptions.
    $eClass = get_class($event->getThrowable());
    if ((!str_starts_with($eClass, 'Drupal\commercetools') && !str_starts_with($eClass, 'Commercetools\\'))
      || $event->getRequest()->getRequestFormat() !== self::REQUEST_FORMAT_COMMERCETOOLS_GRAPHQL) {
      return;
    }

    $e = $event->getThrowable();
    $errorLevel = $this->configFactory->get('system.logging')->get('error_level');

    switch ($errorLevel) {
      case ERROR_REPORTING_DISPLAY_SOME:
        $graphqlError = [
          'message' => $e->getMessage(),
        ];
        break;

      case ERROR_REPORTING_DISPLAY_VERBOSE:
        $graphqlError = [
          'message' => $e->getMessage(),
          "path" => [$e->getFile()],
          "locations" => [
            "line" => $e->getLine(),
          ],
        ];
        break;

      case ERROR_REPORTING_DISPLAY_ALL:
        $graphqlError = [
          'message' => $e->getMessage(),
          "path" => [$e->getFile()],
          "locations" => [
            "line" => $e->getLine(),
          ],
          "extensions" => [
            "code" => "INTERNAL_SERVER_ERROR",
            "stacktrace" => $e->getTrace(),
          ],
        ];
        break;

      case ERROR_REPORTING_HIDE:
        $graphqlError = [
          'message' => $displayMessageText ?? 'The website encountered an unexpected error. Try again later.',
        ];
    }

    $this->logger->error($e);
    $response = new JsonResponse([
      'errors' => [$graphqlError],
    ], Response::HTTP_INTERNAL_SERVER_ERROR);
    $event->setResponse($response);
  }

}
