<?php

namespace Drupal\commercetools_decoupled\Plugin\Block;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Form\FormStateInterface;
use Drupal\commercetools\Plugin\Block\CommercetoolsBlockBase;
use Drupal\commercetools_decoupled\CommercetoolsDecoupled;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the Commercetools Decoupled Cart Summary block.
 *
 * @Block(
 *   id = "commercetools_decoupled_cart_summary",
 *   admin_label = @Translation("Cart Summary"),
 * )
 */
class CommercetoolsDecoupledCartSummaryBlock extends CommercetoolsBlockBase {

  /**
   * Commercetools Decoupled main service.
   *
   * @var \Drupal\commercetools_decoupled\CommercetoolsDecoupled
   */
  protected CommercetoolsDecoupled $ctDecoupled;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create(...func_get_args());
    $instance->ctDecoupled = $container->get('commercetools_decoupled');
    return $instance;
  }

  /**
   * Default number of items to display.
   */
  protected const MAX_ITEMS_DEFAULT = 5;

  /**
   * {@inheritdoc}
   */
  public function blockForm($form, FormStateInterface $form_state) {
    $form = parent::blockForm($form, $form_state);
    $config = $this->getConfiguration();

    $form['max_items'] = [
      '#type' => 'number',
      '#title' => $this->t('Number of items to display'),
      '#description' => $this->t('The number of items to display in the cart summary.'),
      '#default_value' => $config['max_items'] ?? self::MAX_ITEMS_DEFAULT,
      '#min' => 1,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function blockSubmit($form, FormStateInterface $form_state) {
    parent::blockSubmit($form, $form_state);
    $this->setConfigurationValue('max_items', $form_state->getValue('max_items'));
  }

  /**
   * {@inheritdoc}
   */
  public function buildSafe(): array {
    $config = $this->getConfiguration();
    if (empty($config['max_items'])) {
      $config['max_items'] = self::MAX_ITEMS_DEFAULT;
    }
    return $this->ctDecoupled->addDecoupledSettings([
      '#type' => 'component',
      '#component' => 'commercetools_decoupled:cart-block',
      '#props' => $config,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    $cacheContexts = parent::getCacheContexts();
    return Cache::mergeContexts($cacheContexts, [
      'commercetools_cart',
    ]);
  }

}
