<?php

namespace Drupal\commercetools;

use Drupal\commercetools\Form\SubscriptionSettingsForm;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Config\TypedConfigManagerInterface;

/**
 * Commercetools Configuration service.
 */
class CommercetoolsConfiguration {

  const CONFIGURATION_API = 'commercetools.api';
  const CONFIGURATION_SETTINGS = 'commercetools.settings';

  const API_CREDENTIAL_KEYS = [
    'client_id',
    'client_secret',
    'hosted_region',
    'project_key',
  ];

  /**
   * Commercetools connection settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $connection;

  /**
   * Commercetools general settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  public ImmutableConfig $settings;

  /**
   * CommercetoolsApiService constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   */
  public function __construct(
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly TypedConfigManagerInterface $typedConfigManager,
  ) {
    $this->connection = $this->configFactory->get(self::CONFIGURATION_API);
    $this->settings = $this->configFactory->get(self::CONFIGURATION_SETTINGS);
  }

  /**
   * Provides a current commercetools connection configuration.
   *
   * @return array
   *   A full list of settings as associative array.
   */
  public function getConnectionConfig(): array {
    $configKeys = [
      CommercetoolsApiServiceInterface::CONFIG_CLIENT_ID,
      CommercetoolsApiServiceInterface::CONFIG_CLIENT_SECRET,
      CommercetoolsApiServiceInterface::CONFIG_PROJECT_KEY,
      CommercetoolsApiServiceInterface::CONFIG_SCOPE,
      CommercetoolsApiServiceInterface::CONFIG_HOSTED_REGION,
      CommercetoolsApiServiceInterface::CONFIG_CACHE_RESPONSES_TTL,
    ];

    $config = [];
    foreach ($configKeys as $configKey) {
      $config[$configKey] = $this->connection->get($configKey);
    }

    return $config;
  }

  /**
   * Determines whether the current configuration is complete.
   *
   * @param string $configName
   *   The configuration name.
   * @param array $byOptions
   *   The configuration keys list to check by.
   *
   * @return bool
   *   TRUE if all provided configuration options are set, false otherwise.
   */
  public function isConfigured(string $configName, array $byOptions): bool {
    $config = $this->configFactory->get($configName);
    foreach ($byOptions as $key) {
      if (empty($config->get($key))) {
        return FALSE;
      }
    }
    return TRUE;
  }

  /**
   * Provides a list of credential configuration keys.
   *
   * @return array
   *   A list of connection option keys.
   */
  public function listCredentialKeys(): array {
    return self::API_CREDENTIAL_KEYS;
  }

  /**
   * Determines whether the current connection configuration is complete.
   *
   * @param array $byOptions
   *   The configuration keys list to limit the check by. Optional.
   *
   * @return bool
   *   TRUE if all configuration options are set, false otherwise.
   */
  public function isConnectionConfigured(array $byOptions = []): bool {
    $options = $byOptions ?: $this->listCredentialKeys();
    return $this->isConfigured(self::CONFIGURATION_API, $options);
  }

  /**
   * Checks if the provided options matches the configuration.
   *
   * @param array $options
   *   The list of configuration options to check by.
   * @param \Drupal\Core\Config\ImmutableConfig $config
   *   Configuration to check.
   *
   * @return bool
   *   TRUE if all provided options match current ones, false otherwise.
   */
  public function isEqual(array $options, ImmutableConfig $config): bool {
    foreach ($options as $key => $option) {
      $configValue = $config->get($key);
      // Check if an option is set and equal to the one in the configuration.
      if (!isset($configValue) || $configValue != $option) {
        return FALSE;
      }
    }
    return TRUE;
  }

  /**
   * Checks if the connection options matches the current configuration.
   *
   * @param array $options
   *   The list of configuration options to check by.
   *
   * @return bool
   *   TRUE if all provided options match current ones, false otherwise.
   */
  public function isConnectionEqual(array $options): bool {
    return $this->isEqual($options, $this->connection);
  }

  /**
   * Detects switching to a new project by updated configs.
   *
   * @param array $updated
   *   The updated configuration options.
   *
   * @return bool
   *   TRUE if switching to a new project, false otherwise.
   */
  public function isProjectChanging(array $updated): bool {
    if (!$this->isConnectionConfigured()) {
      return FALSE;
    }
    $options = [];
    foreach ($this->listCredentialKeys() as $key) {
      $options[$key] = $updated[$key] ?? '';
    }
    return !$this->isConnectionEqual($options);
  }

  /**
   * Indicates whether subscriptions are configured.
   *
   * @return bool
   *   TRUE if a destination is configured; FALSE otherwise.
   */
  public function isSubscriptionsConfigured(): bool {
    return (bool) $this->configFactory->get(SubscriptionSettingsForm::CONFIGURATION_NAME)
      ->get('destination');
  }

}
