<?php

namespace Drupal\commercetools;

use Drupal\commercetools\EventSubscriber\CommercetoolsSessionIdStorage;
use Drupal\commercetools\Exception\CommercetoolsMissingPermissionException;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\Exception\ConnectException;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * The commercetools session service that interacts with CT Session API.
 */
class CommercetoolsSession {

  /**
   * Configuration settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The CommercetoolsSession constructor.
   *
   * @param \Drupal\commercetools\CommercetoolsApiServiceInterface $ctApi
   *   The Commercetools API service.
   * @param \Drupal\commercetools\CommercetoolsCarts $ctCart
   *   The commercetools cart service.
   * @param \Drupal\commercetools\EventSubscriber\CommercetoolsSessionIdStorage $ctSessionIdStorage
   *   The Commercetools session ID storage service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    protected readonly CommercetoolsApiServiceInterface $ctApi,
    protected readonly CommercetoolsCarts $ctCart,
    protected readonly CommercetoolsSessionIdStorage $ctSessionIdStorage,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly LoggerInterface $logger,
  ) {
    $this->config = $this->configFactory->get(CommercetoolsApiServiceInterface::CONFIGURATION_NAME);
  }

  /**
   * Resolves a checkout session by Application key.
   *
   * @param string $applicationKey
   *   The Checkout Application key.
   *
   * @return string
   *   A CT checkout session ID if resolved.
   */
  public function resolveCheckoutSession(string $applicationKey): string {
    // First, look for an active checkout session.
    if ($sessionId = $this->ctSessionIdStorage->getValue()) {
      return $sessionId;
    }

    // If no, create a new checkout session by current cart ID.
    if ($cartId = $this->ctCart->getCurrentCartId()) {
      $host = $this->ctApi->getApiUrl('session');
      $projectKey = $this->config->get(CommercetoolsApiServiceInterface::CONFIG_PROJECT_KEY);
      $url = "{$host}/{$projectKey}/sessions";
      $options = [
        'json' => [
          'cart' => [
            'cartRef' => ['id' => $cartId],
          ],
          'metadata' => [
            'applicationKey' => $applicationKey,
          ],
        ],
      ];
      $client = $this->ctApi->getClient();
      try {
        $response = $client->request('POST', $url, $options);
        $sessionData = Json::decode($response->getBody()->getContents());
        $sessionId = $sessionData['id'];
        $this->ctSessionIdStorage->setValue($sessionId);
        $this->ctSessionIdStorage->setExpireTime($sessionData['expiryAt']);
        return $sessionId;
      }
      catch (ConnectException | GuzzleException $e) {
        if ($e->getCode() === Response::HTTP_FORBIDDEN) {
          throw new CommercetoolsMissingPermissionException('The commercetools session API request failed with the forbidden error.', $e->getCode(), $e);
        }
        throw $e;
      }
    }

    return '';
  }

}
