<?php

namespace Drupal\commercetools;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drush\Drush;

/**
 * Contains implementations of commercetools historical updates.
 *
 * @see commercetools.api.php
 */
class CommercetoolsUpdates {

  /**
   * Saves the new values to the configuration keys.
   *
   * @param array $values
   *   A key-value array of values.
   * @param string $name
   *   The configuration key name. The commercetools settings key by default.
   */
  public static function setConfigKeys(
    array $values,
    string $name = CommercetoolsService::CONFIGURATION_NAME,
  ): void {
    $config_factory = \Drupal::configFactory();
    $config = $config_factory->getEditable($name);
    foreach ($values as $key => $value) {
      $config->set($key, $value);
    }
    $config->save(TRUE);
  }

  /**
   * Unsets specific values in a configuration.
   *
   * @param array $keys
   *   The list of configuration option keys to unset.
   * @param string $name
   *   The configuration key name. The commercetools settings key by default.
   */
  public static function clearConfigKeys(
    array $keys,
    string $name = CommercetoolsService::CONFIGURATION_NAME,
  ): void {
    $config_factory = \Drupal::configFactory();
    $config = $config_factory->getEditable($name);
    foreach ($keys as $key) {
      $config->clear($key);
    }
    $config->save(TRUE);
  }

  /**
   * Moves configuration values from one key to another.
   *
   * @param array $fromTo
   *   An array of the config names and keys, with the source key as the key
   *   and the target key as the value.
   *   Example:
   *   - `["my_old_config:my_old_key" => "my_new_config:my_new_key"]`.
   */
  public static function migrateConfigKeys(
    array $fromTo,
  ): void {
    $configFactory = \Drupal::configFactory();
    $configs = [];
    foreach ($fromTo as $from => $to) {
      [$fromName, $fromKey] = explode(':', $from);
      [$toName, $toKey] = explode(':', $to);
      $configs[$fromName] ??= $configFactory->getEditable($fromName);
      $configs[$toName] ??= $configFactory->getEditable($toName);
      $configs[$toName]->set($toKey, $configs[$fromName]->get($fromKey));
      $configs[$fromName]->clear($fromKey);
    }
    foreach ($configs as $config) {
      $config->save(TRUE);
    }
  }

  /**
   * Displays a message to the user using Drush or Drupal messenger.
   *
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup $message
   *   The message to display.
   */
  public static function displayMessage(TranslatableMarkup $message): void {
    if (class_exists(Drush::class) && Drush::hasContainer()) {
      Drush::logger()->notice($message);
    }
    else {
      \Drupal::messenger()->addStatus($message);
    }
  }

}
