<?php

namespace Drupal\commercetools\EventSubscriber;

use Drupal\commercetools\CommercetoolsApiServiceInterface;
use Drupal\commercetools\CommercetoolsCustomers;
use Drupal\commercetools\CommercetoolsLocalization;
use Drupal\commercetools\CommercetoolsService;
use Drupal\commercetools\Event\CommercetoolsConfigurationEvent;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Performs clean-up on main configuration changed event.
 */
class CommercetoolsConfigChangedSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * CommercetoolsConfigChangedSubscriber constructor.
   *
   * @param \Drupal\commercetools\EventSubscriber\CommercetoolsSessionIdStorage $ctSessionIdStorage
   *   The session ID storage.
   * @param \Drupal\commercetools\CommercetoolsCustomers $ctCustomers
   *   The commercetools customers service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \Drupal\commercetools\CommercetoolsApiServiceInterface $ctApi
   *   The commercetools API service.
   * @param \Drupal\commercetools\CommercetoolsLocalization $ctLocalization
   *   The commercetools localization service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   */
  public function __construct(
    protected readonly CommercetoolsSessionIdStorage $ctSessionIdStorage,
    protected readonly CommercetoolsCustomers $ctCustomers,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly CommercetoolsApiServiceInterface $ctApi,
    protected readonly CommercetoolsLocalization $ctLocalization,
    protected readonly MessengerInterface $messenger,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      CommercetoolsConfigurationEvent::class => ['doCleanup'],
    ];
  }

  /**
   * Process the message.
   *
   * @param \Drupal\commercetools\Event\CommercetoolsConfigurationEvent $event
   *   The message event.
   */
  public function doCleanup(CommercetoolsConfigurationEvent $event) {
    $this->ctSessionIdStorage->deleteValue();
    $this->ctCustomers->unsetCustomerIdForUsers();
    $config = $this->configFactory->getEditable(CommercetoolsService::CONFIGURATION_NAME);
    $config->set(CommercetoolsService::CONFIG_PAGE_ATTRIBUTES_ENABLED, FALSE)
      ->set(CommercetoolsService::CONFIG_PAGE_ATTRIBUTES, [])
      ->save();
    try {
      // Reconnect the client with the updated credentials.
      $updatedConfig = $event->getConfig();
      $this->ctApi->setOverriddenConfig($updatedConfig);
      $projectInfo = $this->ctApi->getProjectInfo();
      $this->ctLocalization->completeProjectLocalization($projectInfo);
    }
    catch (\Exception) {
      $this->messenger->addWarning($this->t('Unable to retrieve an API response with current project information. If everything is ok with your Internet connection, review the connection settings once again.'));
    }
    finally {
      $this->ctApi->restoreOriginalConfig();
    }
  }

}
