<?php

namespace Drupal\commercetools\Form;

use Drupal\commercetools\CommercetoolsService;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure a Commercetools settings form for this site.
 */
class AttributesSettingsForm extends CommercetoolsSettingsFormBase {

  /**
   * The Commercetools service.
   *
   * @var \Drupal\commercetools\CommercetoolsService
   */
  protected CommercetoolsService $ct;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->ct = $container->get('commercetools');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $productTypes = $this->ct->getProductsTypes();
    $attributeTypes = [
      "boolean" => $this->t('Yes / No (boolean)'),
      "text" => $this->t('Text'),
      "ltext" => $this->t('Localized Text'),
      "enum" => $this->t('List (enum)'),
      "lenum" => $this->t('Localized List (enum)'),
      "number" => $this->t('Number'),
      "money" => $this->t('Money'),
      "date" => $this->t('Date'),
      "time" => $this->t('Time'),
      "datetime" => $this->t('Date and Time'),
      "reference" => $this->t('Reference'),
      "set" => $this->t('Set'),
      "nested" => $this->t('Nested'),
    ];

    $pageOptions = [];
    $filterOptions = [];
    foreach ($productTypes as $productTypeKey => $productType) {
      foreach ($productType['attributeDefinitions'] as $attributeKey => $attribute) {
        $isFieldTypeSupported = !in_array($attribute['type'], ['enum', 'lenum', 'ltext']) && $attribute['isSearchable'];
        $supportedText = $isFieldTypeSupported ? '' : ', ' . $this->t('not supported');
        $attributeLabel = $attributeTypes[$attribute['type']] ?? $this->t('Other');
        $pageOptions[$productTypeKey . '_' . $attributeKey] = $this->t('@productType: <b>@attrName</b> (@attrType)', [
          '@attrName' => $attribute['label'],
          '@attrType' => $attributeLabel,
          '@productType' => $productType['name'],
        ]);
        $filterOptions[$productTypeKey . '_' . $attributeKey] = $this->t('@productType: <b>@attrName</b> (@attrType)', [
          '@attrName' => $attribute['label'],
          '@attrType' => $attributeLabel . $supportedText,
          '@productType' => $productType['name'],
        ]);
      }
    }

    $fieldsets = [
      'page' => [
        'label' => $this->t('Customize displayed product attributes on the product page'),
        'description' => $this->t('By default, all available product attributes are displayed. Check to customize the displayed attributes.'),
        'enableKey' => CommercetoolsService::CONFIG_PAGE_ATTRIBUTES_ENABLED,
        'attributesKey' => CommercetoolsService::CONFIG_PAGE_ATTRIBUTES,
        'options' => $pageOptions,
      ],
    ];
    foreach ($fieldsets as $fieldsetKey => $fieldset) {
      $form[$fieldsetKey . '_wr'] = [
        '#type' => 'fieldset',
        $fieldset['enableKey'] => [
          '#type' => 'checkbox',
          '#title' => $fieldset['label'],
          '#description' => $fieldset['description'],
          '#config_target' => CommercetoolsService::CONFIGURATION_NAME . ':' . $fieldset['enableKey'],
        ],
        $fieldset['attributesKey'] . '_wr' => [
          '#type' => 'container',
          '#states' => [
            'visible' => [
              ':input[name="' . $fieldset['enableKey'] . '"]' => [
                'checked' => TRUE,
              ],
            ],
          ],
          $fieldset['attributesKey'] => [
            '#type' => 'checkboxes',
            '#options' => $fieldset['options'],
            '#config_target' => CommercetoolsService::CONFIGURATION_NAME . ':' . $fieldset['attributesKey'],
          ],
        ],
      ];
    }

    $form['updateAttributes'] = [
      '#type' => 'submit',
      '#value' => $this->t('Update product types and attributes from Commercetools'),
      '#submit' => ['::updateProductTypes'],
      '#limit_validation_errors' => [],
    ];

    $form = parent::buildForm($form, $form_state);
    return $form;
  }

  /**
   * Submit function to refresh product types.
   */
  public function updateProductTypes(array $form, FormStateInterface $form_state) {
    $this->ct->getProductsTypes(TRUE);
  }

}
