<?php

declare(strict_types=1);

namespace Drupal\commercetools\Plugin\Block;

use Drupal\commercetools\Exception\CommercetoolsGraphqlErrorException;
use Drupal\commercetools\Exception\CommercetoolsOperationFailedException;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a base class for commercetools component blocks.
 */
abstract class CommercetoolsBlockBase extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The service container.
   */
  protected ContainerInterface $container;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
    );
    $instance->container = $container;
    return $instance;
  }

  /**
   * Wraps the build method to display inline errors in block on exceptions.
   *
   * {@inheritdoc}
   */
  public function build() {
    try {
      return $this->buildSafe();
    }
    catch (CommercetoolsOperationFailedException | CommercetoolsGraphqlErrorException $e) {
      $this->container->get('logger.factory')->get('commercetools')->error($e->getMessage());

      return [
        '#theme' => 'status_messages',
        '#message_list' => [
          'error' => [
            $this->t('An error occurred while building the block @block_id: @message', [
              '@block_id' => $this->getPluginId(),
              '@message' => $e->getMessage(),
            ]),
          ],
        ],
        '#attributes' => [
          'class' => [
            'commercetools-block-render-failed',
            'commercetools-block-' . $this->getPluginId(),
          ],
        ],
        '#cache' => [
          'max-age' => 0,
        ],
      ];
    }
  }

  /**
   * A safe build method to handle exceptions as inline errors.
   *
   * @return array
   *   A render array containing the block content.
   */
  abstract public function buildSafe(): array;

}
