<?php

namespace Drupal\commercetools\Routing;

use Drupal\commercetools\Form\UiModuleSettingsFormBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;

/**
 * Defines dynamic routes.
 */
abstract class UiModulesRouteProviderBase implements ContainerInjectionInterface {

  const CONFIGURATION_NAME = NULL;
  const ROUTE_PREFIX = NULL;
  const ROUTE_CONTROLLERS = [];

  const PAGE_CATALOG_ROUTE = 'catalog';
  const PAGE_PRODUCT_ROUTE = 'product_page';
  const PAGE_CART_ROUTE = 'carts';
  const PAGE_CHECKOUT_ROUTE = 'checkout';
  const PAGE_RESET_CART_ROUTE = 'reset_page';
  const PAGE_USER_ORDER_ROUTE = 'user_order';
  const PAGE_USER_ORDERS_ROUTE = 'user_orders';


  /**
   * A config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->configFactory = $container->get('config.factory');
    return $instance;
  }

  /**
   * Returns an array of route objects.
   *
   * @return \Symfony\Component\Routing\RouteCollection
   *   An array of route objects.
   */
  public function routes() {
    $config = $this->configFactory->get(static::CONFIGURATION_NAME);
    $catalogPath = $config->get(UiModuleSettingsFormBase::CONFIG_CATALOG_PATH);
    $routeCollection = new RouteCollection();

    $route = new Route($catalogPath);
    $route->setDefault('_title', 'Catalog');
    $route->setDefault('_controller', static::ROUTE_CONTROLLERS[self::PAGE_CATALOG_ROUTE]);
    $route->setRequirement('_access', 'TRUE');
    $routeCollection->add(static::ROUTE_PREFIX . self::PAGE_CATALOG_ROUTE, $route);

    $cartRoute = static::ROUTE_CONTROLLERS[self::PAGE_CART_ROUTE];
    $route = new Route($config->get('cart_path'));
    $route->setDefault('_title', 'My cart');
    $route->setDefault(strpos($cartRoute, 'Form') === FALSE ? '_controller' : '_form', $cartRoute);
    $route->setRequirement('_access', 'TRUE');
    $routeCollection->add(static::ROUTE_PREFIX . self::PAGE_CART_ROUTE, $route);

    $route = new Route("$catalogPath/{slug}");
    $route->setDefault('_controller', static::ROUTE_CONTROLLERS[self::PAGE_PRODUCT_ROUTE]);
    $route->setRequirement('_access', 'TRUE');
    $route->setRequirement('slug', '.+');
    $routeCollection->add(static::ROUTE_PREFIX . self::PAGE_PRODUCT_ROUTE, $route);

    if ($userOrdersPath = $config->get(UiModuleSettingsFormBase::CONFIG_USER_ORDERS_PATH)) {
      $route = new Route("$userOrdersPath/{orderId}");
      $route->setDefault('_controller', static::ROUTE_CONTROLLERS[self::PAGE_USER_ORDER_ROUTE]);
      $route->setRequirement('_access', 'TRUE');
      $route->setRequirement('{user}', '\d+');
      $route->setRequirement('{orderId}', '.+');
      $routeCollection->add(static::ROUTE_PREFIX . self::PAGE_USER_ORDER_ROUTE, $route);

      $route = new Route($userOrdersPath);
      $route->setDefault('_title', 'Orders');
      $route->setDefault('_controller', static::ROUTE_CONTROLLERS[self::PAGE_USER_ORDERS_ROUTE]);
      $route->setRequirement('_permission', 'view own commercetools orders');
      $route->setRequirement('user', '\d+');
      $routeCollection->add(static::ROUTE_PREFIX . self::PAGE_USER_ORDERS_ROUTE, $route);
    }

    $checkoutPath = $config->get(UiModuleSettingsFormBase::CONFIG_CHECKOUT_PATH);
    if ($checkoutPath) {
      $route = new Route($checkoutPath);
      $route->setDefault('_title', 'Checkout');
      $route->setDefault('_controller', static::ROUTE_CONTROLLERS[self::PAGE_CHECKOUT_ROUTE]);
      $route->setRequirement('_access', 'TRUE');
      $routeCollection->add(static::ROUTE_PREFIX . self::PAGE_CHECKOUT_ROUTE, $route);

      $route = new Route($checkoutPath . '/reset-cart');
      $route->setDefault('_controller', static::ROUTE_CONTROLLERS[self::PAGE_RESET_CART_ROUTE]);
      $route->setRequirement('_access', 'TRUE');
      $routeCollection->add(static::ROUTE_PREFIX . self::PAGE_RESET_CART_ROUTE, $route);
    }

    return $routeCollection;
  }

}
