<?php

declare(strict_types=1);

namespace Drupal\commercetools_test\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\State\StateInterface;
use Drupal\test_helpers\Stub\HttpClientFactoryStub;
use GuzzleHttp\Psr7\Response;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Defines a controller for the Commercetools Test pages.
 */
class CommerceToolsTestController extends ControllerBase {

  /**
   * The service container.
   *
   * @var \Symfony\Component\DependencyInjection\ContainerInterface
   */
  protected ContainerInterface $container;

  /**
   * CommerceToolsTestController constructor.
   *
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\test_helpers\Stub\HttpClientFactoryStub $httpClientFactory
   *   The http_client_factory stub.
   */
  public function __construct(
    protected StateInterface $state,
    protected HttpClientFactoryStub $httpClientFactory,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static(
      $container->get('state'),
      $container->get('commercetools_test.http_client_factory_mock'),
    );
    $instance->container = $container;
    return $instance;
  }

  /**
   * Performs the generation and processing of a product update message.
   *
   * @param string $id
   *   The product id.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response with the operation result.
   */
  public function messageInvalidateProduct(string $id): JsonResponse {
    $message = [
      'type' => 'ProductPublished',
      'createdAt' => date('Y-m-d\TH:i:s.Z\Z', time()),
      'resourceRef' => [
        'id' => $id,
        'typeId' => 'product',
      ],
    ];
    $graphQlResponseBody = [
      'data' => [
        'messages' => [
          'total' => 1,
          'results' => [$message],
        ],
      ],
    ];

    $graphQlResponse = new Response(200, [], json_encode($graphQlResponseBody));

    $this->httpClientFactory->stubAddCustomResponseToStack($graphQlResponse);

    // Call the cron hook for the main module.
    commercetools_cron();

    $result = ['status' => 'success'];
    return new JsonResponse($result);
  }

  /**
   * Resets the Commercetools connection to drop auth cache.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response with the operation result.
   */
  public function resetCtConnection(): JsonResponse {
    $this->container->get('commercetools.api')->resetConnection();
    $result = ['status' => 'success'];
    return new JsonResponse($result);
  }

}
