<?php

/**
 * @file
 * Utility functions to include in scripts.
 */

declare(strict_types=1);

/**
 * The commercetools module directory absolute path.
 *
 * @var string
 */
$moduleDirectory = realpath(dirname(__FILE__) . '/../..');

/**
 * The commercetools_test assets directory absolute path.
 *
 * @var string
 */
$assetsDirectory = $moduleDirectory . '/tests/modules/commercetools_test/assets';

/**
 * Read the log file with the test helpers log entries.
 *
 * @param string $path
 *   The path to the log file.
 *
 * @return array
 *   The list of log entries.
 */
function read_test_helpers_log(string $path): array {
  $log = file_get_contents($path);
  $lines = explode("\n", $log);
  $entries = [];
  foreach ($lines as $line) {
    if ($line) {
      $entries[] = json_decode($line, TRUE);
    }
  }
  return $entries;
}

/**
 * Gets the unique sorted list of hashes from the log file.
 *
 * @param string $path
 *   A path to the log file.
 *
 * @return array
 *   The sorted list of used hashes.
 */
function get_used_hashes_from_test_helpers_log(string $path): array {
  $logs = read_test_helpers_log($path);
  $usedHashes = array_values(array_unique(array_column($logs, 'hash')));
  sort($usedHashes);
  return $usedHashes;
}

/**
 * Get the list of stored assets hashes.
 *
 * @param string $dir
 *   The directory to search for the stored assets.
 */
function parse_stored_assets_files($dir) {
  $files = array_diff(scandir($dir), ['..', '.']);
  $otherFiles = [];
  $hashes = array_map(function ($path) use (&$otherFiles) {
    $name = basename($path);
    if (!preg_match('/^([a-f0-9]{32})(\.txt|\.metadata\.yml)$/', $name, $matches)) {
      $otherFiles[] = $name;
      return;
    }
    return $matches[1];
  }, $files);

  $storedHashes = array_values(array_filter($hashes));
  sort($storedHashes);
  $storedHashes = array_unique($storedHashes);

  return [
    'hashes' => $storedHashes,
    'otherFiles' => $otherFiles,
  ];
}

/**
 * Searches for the target file in the current directory and its parents.
 *
 * @return string
 *   The path to the Drupal root directory.
 */
function find_drupal_root() {
  $dir = realpath(dirname(__FILE__));
  $target = 'core/lib/Drupal.php';
  while (TRUE) {
    if (file_exists($dir . '/' . $target)) {
      return $dir;
    }
    $parentDir = dirname($dir);
    if ($dir === $parentDir) {
      break;
    }
    $dir = $parentDir;
  }

  throw new Exception("Can't find Drupal root directory (checking the file '$target') from the directory $dir");
}
