const commerceToolsConfigs = require('./commerceToolsConfigs');
const waitForPlaceholderToDisappear = require('./utils/waitForPlaceholderToDisappear');

function getPath(module) {
  return commerceToolsConfigs[module][`${module}.settings`].catalog_path;
}

let configOriginal;

const pageNotFoundMessage = 'Page not found';

const userRoles = ['anonymous', 'privileged'];

async function checkCommercetoolsConnectionBrokenError(browser, catalogPath) {
  const expectedMessage = 'The commercetools API operation failed: ';
  let commercetoolsApiConfig;
  browser
    // Check the error message with wrong endpoint.
    .perform(() => {
      browser.thGetConfig('commercetools.api', (result) => {
        commercetoolsApiConfig = result.value;
      });
    })
    .thSetConfig('commercetools.api', { hosted_region: 'localhost/' })
    .drupalRelativeURL(catalogPath)
    .assert.textContains('body', expectedMessage)
    // Restore the connection settings to the original state.
    .perform(() => {
      browser.thSetConfig('commercetools.api', commercetoolsApiConfig);
    });
}

function checkProductListNothingFoundError(browser, catalogPath) {
  browser
    // For pages with pager > 1 the site should show the page not found
    // error instead of the nothing found error.
    .drupalRelativeURL(`${catalogPath}?page=999`)
    .perform(waitForPlaceholderToDisappear(browser))
    .assert.textContains('body', pageNotFoundMessage);
}

function checkProductNotFoundError(browser, catalogPath) {
  browser
    .drupalRelativeURL(`${catalogPath}/not-existent-product`)
    .perform(waitForPlaceholderToDisappear(browser))
    .assert.textContains('body', pageNotFoundMessage);
}

function checkNoSlugError(browser, catalogPath) {
  const missingSlugMessage = 'The product slug value is missing for a product';
  let localeConfig;
  browser
    .perform(async () => {
      localeConfig = await browser.thGetConfig('commercetools.locale');
      browser.thSetConfig('commercetools.locale', {
        language_fallback: 'non-exist2',
        language: 'non-exist1',
      });
    })
    .drupalRelativeURL(`${catalogPath}`)
    .perform(waitForPlaceholderToDisappear(browser))
    .assert.textContains('body', missingSlugMessage)
    .perform(() => {
      browser.thSetConfig('commercetools.locale', localeConfig);
    });
}

function checkItemsPerPageError(browser, catalogPath) {
  browser
    .thSetConfig('commercetools.settings', { items_per_page: -1 })
    .drupalRelativeURL(catalogPath)
    .assert.textContains('body', 'Invalid configuration for items per page.')
    .thSetConfig('commercetools.settings', configOriginal);
}

function checkBootstrapError(browser, catalogPath) {
  const expectedMessage =
    'This template requires a Bootstrap library for the design. Please install any Bootstrap-compatible Drupal theme or override the template using any other CSS framework.';
  browser
    .drupalRelativeURL(catalogPath)
    .execute(() => {
      const cssLinks = document.querySelectorAll('link[rel="stylesheet"]');
      cssLinks.forEach((link) => {
        link.parentNode.removeChild(link);
      });
    })
    .assert.textContains('body', expectedMessage);
}

module.exports = function checkErrors(browser, module) {
  browser.thGetConfig('commercetools.settings', (result) => {
    configOriginal = result.value;
    const catalogPath = getPath(module);
    checkProductNotFoundError(browser, catalogPath);
    checkItemsPerPageError(browser, catalogPath);
    checkProductListNothingFoundError(browser, catalogPath);
    checkNoSlugError(browser, catalogPath);
    checkBootstrapError(browser, catalogPath);

    userRoles.forEach((userRole) => {
      checkCommercetoolsConnectionBrokenError(browser, catalogPath, userRole);
    });
  });
};
