const {
  product: { catalog, card, page },
  page: { messages },
  cart,
  bootstrap: { pager },
  product,
} = require('./selectors');
const checkSearchForm = require('./checkSearchForm');
const checkCartSummaryBlock = require('./checkCartSummaryBlock');
const checkProductList = require('./checkProductList');
const getResponseFromLatestByDataType = require('./getResponseFromLatestByDataType');
const checkInStockFilter = require('./checkInStockFilter');
const checkCategoriesList = require('./checkCategoriesList');
const checkFacets = require('./checkFacets');
const checkSubscriptionsWebhook = require('./checkSubscriptionsWebhook');

module.exports = function shopSmokeTest(browser, catalogUrl) {
  browser.perform(() => {
    checkSubscriptionsWebhook(browser);
  });

  browser.perform(async () => {
    browser.globals.ct.settings = await browser.thGetConfig(
      'commercetools.settings',
    );
    browser.globals.ct.locale = await browser.thGetConfig(
      'commercetools.settings',
    );
  });

  const currencyCountryPairs = [
    { currency: 'USD', country: 'US' },
    { currency: 'EUR', country: 'DE' },
  ];
  for (let i = 0; i < currencyCountryPairs.length; i++) {
    const settings = currencyCountryPairs[i];

    let productTitle;

    browser
      .thSetConfig('commercetools.locale', settings)
      // Open the main product catalog page.
      .drupalRelativeURL(catalogUrl)
      .waitForElementVisible(catalog)
      .waitForElementVisible(card.component)
      .perform(async () => {
        checkInStockFilter(browser);
      })

      .perform(async () => {
        const apiResponse = await getResponseFromLatestByDataType(
          browser,
          'productProjectionSearch',
        );
        const productListItems =
          apiResponse.value.data.productProjectionSearch.results;

        await checkFacets(browser, productListItems);
        await checkSearchForm(browser, productListItems, settings);
        await checkProductList(browser, productListItems, settings);
        productTitle = await browser.getText(
          browser.element({ selector: card.component }).findElement(card.title),
        );
        await checkCartSummaryBlock(
          browser,
          productListItems,
          catalogUrl,
          settings,
        );
        await browser
          .drupalRelativeURL(catalogUrl)
          .waitForElementVisible(product.card.component);
      })
      .perform(() => {
        checkCategoriesList(browser, catalogUrl);
        checkInStockFilter(browser);
      })

      // Jump to the second page in the pager and wait for the component to refresh.
      .thPerformAndWaitForReRender(
        (browser) => {
          browser.click({ selector: pager.item, index: 0 });
        },
        { selector: card.component },
      )

      // Check that the product title is changed.
      .perform(async () => {
        const cardComponentElement = browser.element({
          selector: card.component,
        });

        const productTitleCurrent = await browser.getText(
          cardComponentElement.findElement(card.title),
        );
        await browser.assert.notEqual(
          productTitleCurrent,
          productTitle,
          'The product title has not changed after jumping to the page 2 by the pager.',
        );
      })

      // Jump back to the first page in the pager and wait for the component to refresh.
      .thPerformAndWaitForReRender(
        (browser) => {
          browser.click({ selector: pager.item, index: 0 });
        },
        { selector: card.component },
      )

      // Get the product title to check on next pages.
      .perform(async () => {
        const productTitleCurrent = await browser.getText(
          browser.element({ selector: card.component }).findElement(card.title),
        );
        await browser.assert.equal(
          productTitleCurrent,
          productTitle,
          'The product title should be the same after jumping to the page 1 by the pager.',
        );
      })

      // Click on the product card.
      .click(
        browser.element({ selector: card.component }).findElement(card.title),
      )
      .waitForElementVisible(page.component)
      .perform(() => {
        // @todo Add selector for the specific cart line item.
        browser.expect.element(page.title).text.to.contain(productTitle);
      });

    // Cart is implemented only in the coupled module.
    // @todo Fix cart currency issues and enable this check for both modules.
    if (browser.globals.ct.testingModuleName === 'off') {
      browser
        .click(browser.element(page.component).findElement(page.addToCart))

        // Click on the message link.
        .waitForElementVisible(page.component)
        .waitForElementVisible(messages)
        .click(browser.element(messages).findElement('a'))

        // Check the cart contents.
        .waitForElementVisible(cart.lineItems)
        .perform(() => {
          // @todo Add selector for the specific cart line item.
          browser.expect.element(cart.lineItems).text.to.contain(productTitle);
        });
    }
  }
};
