<?php

namespace Drupal\commercetools_content\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\commercetools\CommercetoolsApiServiceInterface;
use Drupal\commercetools\CommercetoolsCarts;
use Drupal\commercetools\CommercetoolsLocalization;
use Drupal\commercetools\CommercetoolsService;
use Drupal\commercetools\CommercetoolsSession;
use Drupal\commercetools\EventSubscriber\CommercetoolsSessionIdStorage;
use Drupal\commercetools\Form\GeneralSettingsForm;
use Drupal\commercetools\Form\UiModuleSettingsFormBase;
use Drupal\commercetools\Routing\UiModulesRouteProviderBase;
use Drupal\commercetools_content\Form\OrderSubmissionForm;
use Drupal\commercetools_content\Routing\RouteProvider;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Returns response for Commercetools Checkout route.
 */
class CommercetoolsCheckoutController extends ControllerBase {

  /**
   * The CT Session API service.
   *
   * @var \Drupal\commercetools\CommercetoolsSession
   */
  protected $ctSession;

  /**
   * The Commercetools service.
   *
   * @var \Drupal\commercetools\CommercetoolsService
   */
  protected $ct;

  /**
   * The Commercetools API service.
   *
   * @var \Drupal\commercetools\CommercetoolsApiServiceInterface
   */
  protected $ctApi;

  /**
   * The commercetools session ID storage.
   *
   * @var \Drupal\commercetools\EventSubscriber\CommercetoolsSessionIdStorage
   */
  protected $ctSessionIdStorage;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;


  /**
   * The commercetools cart service.
   *
   * @var \Drupal\commercetools\CommercetoolsCarts
   */
  protected CommercetoolsCarts $ctCart;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->ctSession = $container->get(CommercetoolsSession::class);
    $instance->ct = $container->get('commercetools');
    $instance->ctApi = $container->get('commercetools.api');
    $instance->ctSessionIdStorage = $container->get(CommercetoolsSessionIdStorage::class);
    $instance->ctCart = $container->get('commercetools.carts');
    $instance->formBuilder = $container->get('form_builder');
    return $instance;
  }

  /**
   * Builds the response.
   */
  public function page(): array|RedirectResponse {
    $build = [
      '#cache' => ['max-age' => 0],
    ];

    $cart = $this->ctCart->getCurrentCart()->getData();
    if (empty($cart['lineItems'])) {
      $this->messenger()->addError($this->t('The cart is empty.'));
      return $build;
    }

    $config = $this->config(CommercetoolsService::CONFIGURATION_NAME);
    $mode = $config->get(CommercetoolsService::CONFIG_CHECKOUT_MODE);

    try {
      $build += $mode === GeneralSettingsForm::CHECKOUT_MODE_LOCAL ? $this->localView() : $this->commercetoolsView();
    }
    catch (\RuntimeException $e) {
      $this->messenger()->addError($e->getMessage());
      return new RedirectResponse(Url::fromRoute(RouteProvider::PAGE_CART_ROUTE)->setAbsolute()->toString());
    }

    return $build;
  }

  /**
   * Return local checkout form.
   */
  public function localView(): array {
    return $this->formBuilder->getForm(OrderSubmissionForm::class);
  }

  /**
   * Render commercetools checkout form.
   */
  protected function commercetoolsView(): array {
    $config = $this->config(CommercetoolsService::CONFIGURATION_NAME);
    $configApi = $this->config(CommercetoolsApiServiceInterface::CONFIGURATION_NAME);
    $configLocale = $this->config(CommercetoolsLocalization::CONFIGURATION_NAME);

    if (!($applicationKey = $config->get(CommercetoolsService::CONFIG_CHECKOUT_CT_APP_KEY))) {
      throw new \RuntimeException('The commercetools Application Key is not configured in the commercetools settings');
    }

    if (!($sessionId = $this->ctSession->resolveCheckoutSession($applicationKey))) {
      throw new \RuntimeException('Something went wrong. Try again later.');
    }

    $build = [];
    $build['wrapper'] = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#value' => $this->t('Initialization... Please, wait a bit.'),
      '#attributes' => [
        'class' => ['ctc-wrapper'],
      ],
    ];
    if ($config->get(CommercetoolsService::CONFIG_CHECKOUT_CT_INLINE)) {
      $build['wrapper']['#attributes']['data-ctc'] = '';
    }
    $build['#attached']['library'][] = 'commercetools_content/checkout';
    $build['#attached']['drupalSettings'] = [
      'commercetoolsContentCheckout' => [
        'projectKey' => $configApi->get(CommercetoolsApiServiceInterface::CONFIG_PROJECT_KEY),
        'apiRegion' => $this->ctApi->getApiUrl('region'),
        'sessionId' => $sessionId,
        'locale' => $configLocale->get(CommercetoolsLocalization::CONFIG_LANGUAGE),
        'cartUri' => $this->config(UiModuleSettingsFormBase::CONFIGURATION_NAME)->get(UiModuleSettingsFormBase::CONFIG_CART_PATH),
        'catalogUri' => $this->config(UiModuleSettingsFormBase::CONFIGURATION_NAME)->get(UiModuleSettingsFormBase::CONFIG_CATALOG_PATH),
        'resetUri' => Url::fromRoute(RouteProvider::ROUTE_PREFIX . UiModulesRouteProviderBase::PAGE_RESET_CART_ROUTE)->toString(),
        'sessionCookieName' => $this->ctSessionIdStorage->getCookieName(),
        'checkoutCompleteUri' => Url::fromRoute('commercetools.checkout_complete')->toString(),
        'orderPathPrefix' => Url::fromRoute(
          RouteProvider::ROUTE_PREFIX . UiModulesRouteProviderBase::PAGE_USER_ORDERS_ROUTE,
          ['user' => $this->currentUser()->id()],
        )->toString(),
      ],
    ];
    return $build;
  }

  /**
   * The page callback function for deleting the user's current cart.
   */
  public function resetCart() {
    $this->ctCart->deleteCurrentCart();
    return ['#cache' => ['max-age' => 0]];
  }

}
