class FiltersForm extends HTMLElement {
  constructor() {
    super();
    // Debounce delay constant
    this.autosubmitDebounceDelay = 500;
  }

  connectedCallback() {
    this.paramName = window.commercetools.getParamNameByIndex(
      'filters',
      this.productListIndex,
    );

    this.queryParams = window.commercetools.getRequestQueryParams(
      this.productListIndex,
    );

    const form = document.createElement('form');
    form.classList.add('filters-form', 'commercetools-filters-form');
    form.innerHTML = '';
    if (this.filterSubmitHandler) {
      form.addEventListener('submit', this.filterSubmitHandler);
    }

    try {
      this.renderFilters(form);
    } catch (error) {
      new Drupal.Message().add(Drupal.t('Failed to load filters.'), {
        type: 'error',
      });
      throw error;
    }
    this.form = form;
    this.appendChild(form);
  }

  renderFilters(form) {
    if (this.isLoading) {
      form.insertAdjacentHTML('beforeend', FiltersForm.placeholder());
      return;
    }
    if (!this.filters.length) {
      return;
    }

    this.filters.forEach((filterObj) => {
      let filter;
      switch (filterObj.widget_type) {
        case 'facet':
        case 'facet_count':
          filter = this.buildFacetFilter(filterObj);
          break;

        case 'textfield':
        case 'checkbox':
          filter = this.buildSimpleFilter(filterObj);
          break;

        case 'separator':
          filter = FiltersForm.buildSeparator(filterObj);
          break;

        case 'custom':
          // @todo Add handling for other custom filters or rename the group
          // from 'custom' to 'range' if only range filters are expected.
          filter = this.buildRangeFilter(filterObj);
          break;
      }
      if (filter) {
        form.append(filter);
      }
    });

    // Add sort select.
    let selectedValue = '';
    if (this.queryParams?.sorts?.originalValues?.length > 0) {
      selectedValue = this.queryParams.sorts.originalValues[0];
    }
    form.insertAdjacentHTML(
      'beforeend',
      FiltersForm.selectContainer(selectedValue),
    );

    // Always render Apply button, but hide it when autosubmit is enabled
    const applyButton = document.createElement('input');
    applyButton.value = Drupal.t('Apply');
    applyButton.type = 'submit';
    applyButton.classList.add(
      'btn-lg',
      'button',
      'js-form-submit',
      'form-submit',
      'btn',
      'btn-primary',
    );

    // Hide button when autosubmit is enabled
    if (this.useAjaxAutosubmit) {
      applyButton.classList.add('js-hide');
    }

    form.append(applyButton);

    // Attach autosubmit handlers if enabled
    if (this.useAjaxAutosubmit) {
      this.attachAutoSubmitHandlers(form);
    }
  }

  buildRangeFilter(filterObj) {
    // Calculate step based on fractionDigits.
    const step = this.fractionDigits > 0 ? 1 / 10 ** this.fractionDigits : 1;

    // Convert stored cents back to display format for input values.
    const fromValue = this.activeFilters?.[filterObj.path]?.from
      ? window.commercetools.convertPrice(
          this.activeFilters[filterObj.path].from,
          this.fractionDigits,
          'divide',
        )
      : '';

    const toValue = this.activeFilters?.[filterObj.path]?.to
      ? window.commercetools.convertPrice(
          this.activeFilters[filterObj.path].to,
          this.fractionDigits,
          'divide',
        )
      : '';

    // Extract min/max prices from facets for placeholders.
    let minPlaceholder = '';
    let maxPlaceholder = '';

    if (this.response?.facets) {
      const priceFacet = this.response.facets.find(
        (facet) => facet.facet === filterObj.path,
      );

      if (priceFacet?.value?.ranges?.length > 0) {
        const priceRange = priceFacet.value.ranges[0];

        // Convert min/max from cents to display format
        minPlaceholder = window.commercetools.convertPrice(
          priceRange.min,
          this.fractionDigits,
          'divide',
        );
        maxPlaceholder = window.commercetools.convertPrice(
          priceRange.max,
          this.fractionDigits,
          'divide',
        );
      }
    }

    const container = document.createElement('fieldset');
    container.className = `fieldgroup form-composite form-item form-wrapper`;
    container.innerHTML = `
      <legend>
        <span class="fieldset-legend">${Drupal.t('Price ranges')}</span>
      </legend>
      <div class="js-form-item js-form-type-number form-type-number js-form-item-from form-item-from mb-3">
        <label for="edit-from--2">${Drupal.t('From:')}</label>
        <input data-drupal-selector="edit-from" type="number" id="edit-from--2" name="${this.paramName}[${filterObj.path}][from]"
          value="${fromValue}" step="${step}" min="0" placeholder="${minPlaceholder}" class="form-number form-control">
      </div>
      <div class="js-form-item js-form-type-number form-type-number js-form-item-to form-item-to mb-3">
        <label for="edit-from--2">${Drupal.t('To:')}</label>
        <input data-drupal-selector="edit-to" type="number" id="edit-to--2" name="${this.paramName}[${filterObj.path}][to]"
          value="${toValue}" step="${step}" min="0" placeholder="${maxPlaceholder}" class="form-number form-control">
        </div>
    `;

    return container;
  }

  static selectContainer(selectedValue = null) {
    const locale = drupalSettings.commercetoolsDecoupled.locale || '';

    const sortOptions = [
      { value: '', label: '- Default -' },
      { value: `name.${locale} asc`, label: 'Name 🡑' },
      { value: `name.${locale} desc`, label: 'Name 🡓' },
      { value: 'price asc', label: 'Price 🡑' },
      { value: 'price desc', label: 'Price 🡓' },
    ];

    return `
      <div class="js-form-item js-form-type-select select mb-3 js-form-item-sorts">
        <label for="edit-sort-by">${Drupal.t('Sort by')}</label>
        <select data-drupal-selector="edit-sort-by" id="edit-sort-by" name="sorts" class="form-select">
          ${sortOptions
            .map((optionConfig) => {
              return `<option value="${optionConfig.value}" ${optionConfig.value === selectedValue ? 'selected="selected"' : null}>${Drupal.t(optionConfig.label)}</option>`;
            })
            .join('')}
        </select>
      </div>
    `;
  }

  static buildSeparator(filterObj) {
    const separator = document.createElement('h3');
    separator.innerHTML = filterObj.label;
    return separator;
  }

  buildSimpleFilter(filterObj) {
    const fClass = filterObj.path
      .replace(/\./g, '-')
      .replace(/[A-Z]+/g, '-$&')
      .toLowerCase();
    const id = `${this.paramName}-${fClass}`;
    const type = filterObj.widget_type;

    const wrapper = document.createElement('div');
    wrapper.className = `js-form-item js-form-type-${type} ${type} mb-3 ${id} js-form-item-${id} form-item-filters-${fClass}`;

    const label = document.createElement('label');
    label.className = 'form-check-label';
    label.setAttribute('for', `edit-${id}`);
    label.innerHTML = `<span>${filterObj.label}</span>`;

    const inputName = `${this.paramName}[${filterObj.path}]`;
    const input = document.createElement('input');
    input.name = inputName;
    input.id = `edit-${id}`;
    if (type === 'checkbox') {
      input.type = 'checkbox';
      input.value = '1';
      input.className = 'form-checkbox form-check-input';
      input.checked = this.activeFilters?.[filterObj.path]?.[0];
      wrapper.appendChild(input);
      wrapper.appendChild(label);
      wrapper.classList.add('form-check');
    } else {
      input.type = 'text';
      input.value = this.activeFilters?.[filterObj.path]?.[0] || '';
      input.size = 60;
      input.maxLength = 128;
      input.className = 'form-control';
      wrapper.appendChild(label);
      wrapper.appendChild(input);
    }

    return wrapper;
  }

  buildFacetFilter(filterObj) {
    const facetObj = this.response.facets.find(
      (facet) => facet.facet === filterObj.path,
    );

    if (!facetObj || !facetObj.value) {
      new Drupal.Message().add(Drupal.t('Invalid filter data.'), {
        type: 'warning',
      });
      return;
    }

    // Hide the empty facet.
    if (!facetObj.value.terms.length) {
      return;
    }

    const facetId = `${this.paramName}-${facetObj.facet}`
      .replace(/\W/g, '-')
      .toLowerCase();

    const fieldset = document.createElement('fieldset');
    fieldset.classList.add(
      'fieldgroup',
      'form-composite',
      'form-item',
      'form-wrapper',
      facetId,
    );
    const legend = document.createElement('legend');
    legend.innerHTML = `<span class="fieldset-legend">${filterObj.label}</span>`;
    fieldset.appendChild(legend);

    facetObj.value.terms.forEach((termObj) => {
      const { term, productCount } = termObj;
      const id = `${facetObj.facet}-${term}`.replace(/\W/g, '-');
      const filterId = `${facetObj.facet}`.replace(/\W/g, '-');

      const checkbox = document.createElement('input');
      checkbox.type = 'checkbox';
      checkbox.name = `${this.paramName}[${facetObj.facet}][]`;
      checkbox.value = term;
      checkbox.checked = this.activeFilters[facetObj.facet]?.includes(term);
      checkbox.setAttribute('id', id);
      checkbox.setAttribute(
        'data-product-attribute-key',
        filterObj.attributeKey,
      );
      checkbox.classList.add('facet-filter', 'form-check-input');

      const label = document.createElement('label');
      label.htmlFor = id;
      label.classList.add('option');
      const eventData = { key: filterObj.attributeKey, labelValue: term };
      document.dispatchEvent(
        new CustomEvent('commercetools:productAttributeLabel', {
          detail: eventData,
        }),
      );
      label.innerHTML =
        filterObj.attributeKey === 'color' ||
        filterObj.attributeKey === 'finish'
          ? eventData.labelValue
          : `<span class="text-truncate">${term}</span>`;
      if (filterObj.widget_type === 'facet_count') {
        label.innerHTML += `<span className="float-end">(${productCount})</span>`;
      }

      const wrapper = document.createElement('div');
      wrapper.className = `js-form-item js-form-type-checkbox checkbox mb-3 js-form-item-${id} form-check`;

      wrapper.appendChild(checkbox);
      wrapper.appendChild(label);
      fieldset.appendChild(wrapper);
    });
    return fieldset;
  }

  static placeholder() {
    return `
      <fieldset class="fieldgroup form-composite js-form-item form-item js-form-wrapper form-wrapper placeholderify">
        <legend><span class="fieldset-legend">Placeholder for the filter name</span></legend>
        <div class="fieldset-wrapper">
          ${Array.from({ length: 6 }, () => {
            return `
              <input type="checkbox" name="name" value="value" class="form-checkbox form-check-input">
              <label for="edit-name" class="option"><span class="ms-3 text-truncate">Filter checkbox</span> <span class="float-end">(10)</span></label>
              <hr />
            `;
          }).join('')}
        </div>
      </fieldset>
    `;
  }

  setLoadingState(state = null) {
    const stateClasses = {
      load: 'placeholderify',
      reload: 'reloadify',
    };

    this.form.classList.remove(...Object.values(stateClasses));
    const stateClass = stateClasses[state];
    if (state !== null && stateClass) {
      this.form.classList.add(stateClass);
    }
  }

  attachAutoSubmitHandlers(form) {
    // Handle checkboxes - submit immediately on change
    const checkboxes = form.querySelectorAll('input[type="checkbox"]');
    checkboxes.forEach((checkbox) => {
      checkbox.addEventListener('change', () => {
        this.triggerSubmit();
      });
    });

    // Handle select elements - submit immediately on change
    const selects = form.querySelectorAll('select');
    selects.forEach((select) => {
      select.addEventListener('change', () => {
        this.triggerSubmit();
      });
    });

    // Handle text/number inputs - submit with debounce
    const textInputs = form.querySelectorAll(
      'input[type="text"], input[type="search"], input[type="number"]',
    );
    let debounceTimer;
    textInputs.forEach((input) => {
      input.addEventListener('input', () => {
        clearTimeout(debounceTimer);
        debounceTimer = setTimeout(() => {
          this.triggerSubmit();
        }, this.autosubmitDebounceDelay);
      });
    });
  }

  triggerSubmit() {
    if (this.filterSubmitHandler && this.form) {
      // Dispatch submit event on the form so currentTarget is set correctly
      const event = new Event('submit', { cancelable: true, bubbles: true });
      this.form.dispatchEvent(event);
    }
  }
}

customElements.define('ct-filters-form', FiltersForm);
