<?php

namespace Drupal\commercetools_decoupled\Controller;

use Drupal\commercetools\CommercetoolsCarts;
use Drupal\commercetools\CommercetoolsService;
use Drupal\commercetools\CommercetoolsSession;
use Drupal\commercetools\EventSubscriber\CommercetoolsSessionIdStorage;
use Drupal\commercetools\Form\GeneralSettingsForm;
use Drupal\commercetools\Routing\UiModulesRouteProviderBase;
use Drupal\commercetools_decoupled\CommercetoolsDecoupled;
use Drupal\commercetools_decoupled\Routing\RouteProvider;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Render\Markup;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Defines a controller for the Commercetools decoupled catalog pages.
 */
class DecoupledPagesController extends ControllerBase {

  /**
   * The CT Session API service.
   *
   * @var \Drupal\commercetools\CommercetoolsSession
   */
  protected $ctSession;

  /**
   * The commercetools session ID storage.
   *
   * @var \Drupal\commercetools\EventSubscriber\CommercetoolsSessionIdStorage
   */
  protected $ctSessionIdStorage;

  /**
   * The commercetools cart service.
   *
   * @var \Drupal\commercetools\CommercetoolsCarts
   */
  protected CommercetoolsCarts $ctCart;

  /**
   * Commercetools Decoupled main service.
   *
   * @var \Drupal\commercetools_decoupled\CommercetoolsDecoupled
   */
  protected CommercetoolsDecoupled $ctDecoupled;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->ctSession = $container->get(CommercetoolsSession::class);
    $instance->ctSessionIdStorage = $container->get(CommercetoolsSessionIdStorage::class);
    $instance->ctCart = $container->get('commercetools.carts');
    $instance->ctDecoupled = $container->get('commercetools_decoupled');
    return $instance;
  }

  /**
   * Displays a page with the list of Commercetools products.
   */
  public function productCatalog(): array {
    return $this->ctDecoupled->addDecoupledSettings([
      [
        '#type' => 'component',
        '#component' => 'commercetools_decoupled:products-list',
        '#props' => [
          'style' => 'cards',
          'product_list_index' => 0,
          'columns_number' => 2,
        ],
      ],
    ]);
  }

  /**
   * Displays a page with a single product.
   *
   * @param string $slug
   *   A slug value of the product.
   *
   * @return array
   *   A render array with the product page.
   */
  public function productPage(string $slug): array {
    return $this->ctDecoupled->addDecoupledSettings([
      // Hiding the title before the element is rendered.
      // @todo Invent a better solution for this.
      '#title' => Markup::create('<span style="display: none">[commercetools_product:name]</span>'),
      [
        '#type' => 'component',
        '#component' => 'commercetools_decoupled:product-page',
        '#props' => [
          'slug' => $slug,
        ],
      ],
    ]);
  }

  /**
   * Displays a page with the Commercetools cart form.
   */
  public function cartPage(): array {
    return $this->ctDecoupled->addDecoupledSettings([
      [
        '#type' => 'component',
        '#component' => 'commercetools_decoupled:cart-page',
      ],
      '#cache' => ['contexts' => ['commercetools_cart']],
    ]);
  }

  /**
   * Displays a page with the Commercetools user orders.
   */
  public function ordersViewPage(): array {
    return $this->ctDecoupled->addDecoupledSettings([
      [
        '#type' => 'component',
        '#component' => 'commercetools_decoupled:orders-page',
      ],
    ]);
  }

  /**
   * Displays a page with the Commercetools user order.
   */
  public function orderViewPage(): array {
    return $this->ctDecoupled->addDecoupledSettings([
      [
        '#type' => 'component',
        '#component' => 'commercetools_decoupled:order-page',
      ],
    ]);
  }

  /**
   * Displays a page with the Commercetools checkout page.
   */
  public function checkoutPage(): array|RedirectResponse {
    $output['wrapper'] = [
      '#type' => 'container',
      'component' => [
        '#type' => 'component',
        '#cache' => [
          'max-age' => 0,
        ],
      ],
    ];

    $settings = $this->config(CommercetoolsService::CONFIGURATION_NAME);
    $mode = $settings->get(CommercetoolsService::CONFIG_CHECKOUT_MODE);

    try {
      if ($mode === GeneralSettingsForm::CHECKOUT_MODE_LOCAL) {
        $output['wrapper']['component'] += [
          '#component' => 'commercetools_decoupled:checkout-form-page',
        ];
      }
      else {
        if (!($applicationKey = $settings->get(CommercetoolsService::CONFIG_CHECKOUT_CT_APP_KEY))) {
          throw new \RuntimeException('The commercetools Application Key is not configured in the commercetools settings');
        }
        if (!($sessionId = $this->ctSession->resolveCheckoutSession($applicationKey))) {
          throw new \RuntimeException('Something went wrong. Try again later.');
        }
        $output['wrapper']['component'] += [
          '#component' => 'commercetools_decoupled:checkout-page',
          '#props' => [
            'sessionId' => $sessionId,
          ],
          '#attached' => [
            'drupalSettings' => [
              'commercetoolsDecoupled' => [
                'sessionCookieName' => $this->ctSessionIdStorage->getCookieName(),
                'checkoutCompleteUri' => Url::fromRoute('commercetools.checkout_complete')->toString(),
              ],
            ],
          ],
        ];

        if ($settings->get(CommercetoolsService::CONFIG_CHECKOUT_CT_INLINE)) {
          $output['wrapper']['#attributes']['data-ctc'] = '';
        }
      }
    }
    catch (\RuntimeException $e) {
      $this->messenger()->addError($e->getMessage());
      return new RedirectResponse(Url::fromRoute(RouteProvider::ROUTE_PREFIX . UiModulesRouteProviderBase::PAGE_CART_ROUTE)->setAbsolute()->toString());
    }

    return $this->ctDecoupled->addDecoupledSettings($output);
  }

  /**
   * The page callback function for deleting the user's current cart.
   */
  public function resetCart() {
    $this->ctCart->deleteCurrentCart();
    return ['#cache' => ['max-age' => 0]];
  }

}
