<?php

namespace Drupal\commercetools;

use Commercetools\Client\ApiRequestBuilder;
use Drupal\commercetools\Cache\CacheableCommercetoolsGraphQlResponse;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\RefinableCacheableDependencyInterface;
use GuzzleHttp\ClientInterface;

/**
 * Interface for the `commercetools.api` service.
 */
interface CommercetoolsApiServiceInterface {

  // @todo Move configuration related constants to CommercetoolsConfiguration.
  const CONFIGURATION_NAME = CommercetoolsConfiguration::CONFIGURATION_API;
  const CONFIG_CLIENT_ID = 'client_id';
  const CONFIG_CLIENT_SECRET = 'client_secret';
  const CONFIG_PROJECT_KEY = 'project_key';
  const CONFIG_SCOPE = 'scope';

  const CONFIG_CACHE_RESPONSES_TTL = 'cache_responses_ttl';
  const CONFIG_HOSTED_REGION = 'hosted_region';

  const API_PROXY_PATH_DEFAULT = '/api/commercetools';

  /**
   * Gets the Guzzle HTTP client with Commercetools middlewares.
   *
   * @return \GuzzleHttp\ClientInterface
   *   The Guzzle Http client.
   */
  public function getClient(): ClientInterface;

  /**
   * Indicates whether an CT API credentials are set.
   */
  public function isAccessConfigured(): bool;

  /**
   * Execute a commercetools GraphQL operation and cache, if enabled.
   *
   * Produces events during the execution:
   * - CommercetoolsGraphQlOperationEvent to alter the request.
   * - CommercetoolsGraphQlOperationResultEvent to alter the response.
   *
   * @param string $query
   *   The GraphQL query as string.
   * @param array|null $variables
   *   The variables to pass to the query.
   * @param \Drupal\Core\Cache\CacheableMetadata|null $cacheMetadata
   *   The cacheable metadata to apply for the current operation.
   *   If not provided, the function will auto-detect a caching strategy
   *   by the common cache TTL value, operation name and variables.
   *   You can customize this cache auto-detecting by subscribing to the
   *   CommercetoolsGraphQlOperationEvent event.
   *
   * @return \Drupal\commercetools\Cache\CacheableCommercetoolsGraphQlResponse
   *   The response object with cacheable metadata.
   *
   * @see \Drupal\commercetools\EventSubscriber\CommercetoolsGraphQlCacheSubscriber
   */
  public function executeGraphQlOperation(
    string $query,
    ?array $variables = NULL,
    ?CacheableMetadata $cacheMetadata = NULL,
  ): CacheableCommercetoolsGraphQlResponse;

  /**
   * Gets the Commercetools API Request Builder.
   *
   * @return \Commercetools\Client\ApiRequestBuilder
   *   The ApiRequestBuilder.
   */
  public function getBuilder(): ApiRequestBuilder;

  /**
   * Prepares a response with cacheable metadata.
   *
   * @param array $data
   *   The data to include in the response.
   * @param \Drupal\Core\Cache\RefinableCacheableDependencyInterface|null $cache
   *   The cacheable metadata.
   *
   * @return \Drupal\commercetools\CacheableCommercetoolsResponse
   *   The prepared response.
   */
  public function prepareResponse(array $data, ?RefinableCacheableDependencyInterface $cache = NULL): CacheableCommercetoolsResponse;

  /**
   * Gets the commercetools API URL information.
   *
   * @param string $type
   *   The type of URL to get.
   *   - region: the commercetools API Region name.
   *   - api: the commercetools API URL.
   *   - auth: the commercetools OAuth API URL.
   *   - session: the commercetools Session API URL.
   *
   * @return string
   *   The Region name or the URL.
   */
  public function getApiUrl(string $type = 'api'): string;

  /**
   * Provides basic info about current commercetools project.
   *
   * It includes:
   * - key
   * - name
   * - countries list
   * - currencies list
   * - languages list
   * - customer groups list
   * - channels list
   * - stores list.
   *
   * @return array
   *   Basic info as associative array.
   */
  public function getProjectInfo(): array;

  /**
   * Return the current connection configuration.
   */
  public function getConnectionConfig(): array;

}
