<?php

namespace Drupal\commercetools;

use GraphQL\Actions\Mutation;
use GraphQL\Actions\Query;
use GraphQL\Entities\Node;
use GraphQL\Entities\Variable;

/**
 * The subscriptions api service.
 */
class CommercetoolsSubscriptionsApi {

  /**
   * CommercetoolsSubscriptionsApi constructor.
   *
   * @param \Drupal\commercetools\CommercetoolsService $ct
   *   The Commercetools service.
   * @param \Drupal\commercetools\CommercetoolsApiServiceInterface $ctApi
   *   The Commercetools API service.
   */
  public function __construct(
    protected CommercetoolsService $ct,
    protected CommercetoolsApiServiceInterface $ctApi,
  ) {
  }

  /**
   * Fetches a subscription by its key.
   *
   * @param string $key
   *   Subscription key.
   *
   * @return array
   *   Subscription data as an associative array.
   */
  public function getSubscriptionByKey(string $key): array {
    $arguments = [
      'key' => new Variable('key', 'String!'),
    ];
    $query = new Query('subscription', $arguments);
    $this->addSubscriptionFields($query);
    $response = $this->ctApi->executeGraphQlOperation($query, ['key' => $key]);
    return (array) $response->getData()['subscription'];
  }

  /**
   * Creates a subscription with the given destination and change topics.
   *
   * @param string $key
   *   Subscription key.
   * @param string $destination
   *   Destination discriminator (e.g., "SQS").
   * @param array $paramsDestination
   *   Destination-specific input payload.
   * @param array $changes
   *   List of resource type IDs to subscribe to.
   *
   * @return array
   *   Created subscription data.
   */
  public function createSubscription(string $key, string $destination, array $paramsDestination, array $changes): array {
    $subscriptionDraft = [
      'key' => $key,
      'destination' => [
        $destination => $paramsDestination,
      ],
    ];

    foreach ($changes as $change) {
      $subscriptionDraft['changes'][] = ['resourceTypeId' => $change];
    }

    $arguments = [
      'draft' => new Variable('draft', 'SubscriptionDraft!'),
    ];
    $subscriptionMutation = new Mutation('createSubscription', $arguments);
    $this->addSubscriptionFields($subscriptionMutation);
    $variables = ['draft' => $subscriptionDraft];
    $response = $this->ctApi->executeGraphQlOperation($subscriptionMutation, $variables);
    return (array) $response->getData()['createSubscription'];
  }

  /**
   * Updates a subscription.
   *
   * @param string $key
   *   Subscription key.
   * @param int $version
   *   Current version.
   * @param array $actions
   *   List of SubscriptionUpdateAction items.
   *
   * @return array
   *   GraphQL response data for the update.
   */
  public function updateSubscription(string $key, int $version, array $actions): array {
    $arguments = [
      'key' => new Variable('key', 'String'),
      'version' => new Variable('version', 'Long!'),
      'actions' => new Variable('actions', '[SubscriptionUpdateAction!]!'),
    ];
    $updateMutation = new Mutation('updateSubscription', $arguments);
    $this->addSubscriptionFields($updateMutation);
    $variables = [
      'key' => $key,
      'version' => $version,
      'actions' => $actions,
    ];
    $response = $this->ctApi->executeGraphQlOperation($updateMutation, $variables);
    return (array) $response->getData()['updateSubscription'];
  }

  /**
   * Deletes a subscription by key and version.
   *
   * @param string $key
   *   Subscription key.
   * @param int $version
   *   Current version.
   *
   * @return array
   *   Deletion result (e.g., deleted entity or identifiers).
   */
  public function deleteSubscription(string $key, int $version): array {
    $arguments = [
      'key' => new Variable('key', 'String'),
      'version' => new Variable('version', 'Long!'),
    ];
    $deleteMutation = new Mutation('deleteSubscription', $arguments);
    $deleteMutation->use('key');
    $variables = [
      'key' => $key,
      'version' => $version,
    ];
    $response = $this->ctApi->executeGraphQlOperation($deleteMutation, $variables);
    return (array) $response->getData()['deleteSubscription'];
  }

  /**
   * Add subscription fields to request node.
   *
   * @param \GraphQL\Entities\Node $subscriptionQuery
   *   The subscription node to which the fields will be added.
   */
  protected function addSubscriptionFields(Node $subscriptionQuery): void {
    $this->ct->addFieldsToNode(
      $subscriptionQuery,
      [
        'id',
        'key',
        'version',
      ],
    );
  }

}
