<?php

namespace Drupal\commercetools;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * The base implementation for a storage service that serves a cookie value.
 */
abstract class CookieStorageBase implements EventSubscriberInterface {

  /**
   * The value that is about to be set.
   *
   * @var mixed
   */
  protected mixed $newValue;

  /**
   * Indicates whether the value should be updated.
   *
   * @var bool
   */
  protected bool $shouldUpdate = FALSE;

  /**
   * Indicates whether the value should be deleted.
   *
   * @var bool
   */
  protected bool $shouldDelete = FALSE;

  /**
   * CookieStorageBase constructor.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   Core request stack.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(
    protected readonly RequestStack $requestStack,
    protected readonly TimeInterface $time,
    protected readonly ConfigFactoryInterface $configFactory,
  ) {}

  /**
   * Returns a stored value.
   *
   * @return mixed
   *   A stored value.
   */
  public function getValue(): mixed {
    // If the value is updated from somewhere else, return updated.
    if (!empty($this->newValue)) {
      return $this->newValue;
    }

    return $this->requestStack->getCurrentRequest()
      ->cookies->get($this->getCookieName());
  }

  /**
   * Sets the new value.
   *
   * @param bool|int|null|string $value
   *   The value to store.
   */
  public function setValue(mixed $value): void {
    $this->shouldUpdate = TRUE;
    $this->newValue = $value;
  }

  /**
   * Marks stored value to be deleted.
   */
  public function deleteValue(): void {
    $this->shouldDelete = TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::RESPONSE => 'onResponse',
    ];
  }

  /**
   * Reacts to the symfony kernel response event.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The event to process.
   */
  public function onResponse(ResponseEvent $event): void {
    $response = $event->getResponse();

    // If the value should be deleted, clear the related cookie and exit.
    if ($this->shouldDelete) {
      $response->headers->clearCookie($this->getCookieName());
      return;
    }

    // If the value should be updated, set a new cookie value.
    if ($this->shouldUpdate) {
      $cookie = Cookie::create(
        $this->getCookieName(),
        $this->getValue(),
        $this->getExpireTime()
      );
      $response->headers->setCookie($cookie);
    }
  }

  /**
   * Returns a value identifier.
   *
   * @return string
   *   A stored value.
   */
  abstract public function getCookieName(): string;

  /**
   * Returns an expected cookie expiration time.
   *
   * @return int
   *   A stored value.
   */
  abstract protected function getExpireTime(): int;

}
