<?php

namespace Drupal\commercetools\Form;

use Drupal\commercetools\CommercetoolsApiServiceInterface;
use Drupal\commercetools\CommercetoolsService;
use Drupal\Core\Config\Schema\Undefined;
use Drupal\Core\Form\ConfigFormBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure a Commercetools settings form for this site.
 */
abstract class CommercetoolsSettingsFormBase extends ConfigFormBase {

  const CONFIGURATION_NAME = CommercetoolsService::CONFIGURATION_NAME;

  /**
   * The Commercetools API service.
   *
   * @var \Drupal\commercetools\CommercetoolsApiServiceInterface
   */
  protected CommercetoolsApiServiceInterface $ctApi;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->ctApi = $container->get('commercetools.api');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    $namespace = explode('\\', get_class($this));
    // Compose the form id by template: `{module_name}_{class_short_name}`.
    return $this->camelCaseToSnakeCase($namespace[1] . '_' . end($namespace));
  }

  /**
   * Converts a camelCase string to snake_case.
   *
   * @param string $input
   *   The input string in camelCase.
   *
   * @return string
   *   The converted string in snake_case.
   */
  private function camelCaseToSnakeCase(string $input): string {
    $pattern = '/([a-z])([A-Z])/';
    $replacement = '\\1_\\2';

    $snake_case = preg_replace($pattern, $replacement, $input);
    return strtolower($snake_case);
  }

  /**
   * {@inheritdoc}
   */
  public function getEditableConfigNames() {
    return [$this::CONFIGURATION_NAME];
  }

  /**
   * Generates a form element for a specific setting.
   *
   * @param string $key
   *   The key of the setting.
   * @param array|null $options
   *   (optional) A list of options for the form element.
   * @param string|null $configName
   *   (optional) A custom config target configuration name.
   *
   * @return array
   *   The form element.
   *
   * @see \Drupal\Core\Form\FormBuilderInterface
   */
  protected function getFormElement(string $key, ?array $options = NULL, $configName = NULL) {
    $options ??= [];
    $configName ??= $this::CONFIGURATION_NAME;
    $configKey = $configName . ':' . $key;
    $element = $options + [
      '#type' => 'textfield',
      '#title' => $this->getSettingLabel($configKey),
      '#config_target' => $configKey,
    ];
    return $element;
  }

  /**
   * Gets a label for a setting from typed settings object.
   *
   * @param string $configKey
   *   The key of the setting.
   *
   * @return string
   *   The label of the setting.
   */
  protected function getSettingLabel(string $configKey): string {
    try {
      [$name, $key] = explode(':', $configKey);
      $setting = $this->typedConfigManager->get($name)->get($key);
      if ($setting instanceof Undefined) {
        throw new \InvalidArgumentException('Undefined key in schema');
      }
      $label = $setting->getDataDefinition()->getLabel();
    }
    catch (\InvalidArgumentException) {
      $label = $key;
    }
    return $label;
  }

}
