const {
  product: { page },
  page: { messages },
} = require('./selectors');
const getSomeProduct = require('./getSomeProduct');
const getResponseFromLatestByDataType = require('./getResponseFromLatestByDataType');
const setMockMode = require('./setMockMode');
const unsetMockMode = require('./unsetMockMode');
const commerceToolsConfigs = require('./commerceToolsConfigs');

module.exports = async function checkProductPageCaching(browser, module) {
  const urlPrefix =
    commerceToolsConfigs[module][`${module}.settings`].catalog_path;
  const titleMocked = 'Product Title Mocked';
  const titleMocked2 = 'Product Title2 Mocked';
  const dataType = 'products';

  let product;
  let titleFromApi;
  let url;

  await getSomeProduct(browser, urlPrefix, (result) => {
    product = result.value;
    titleFromApi = product.name;
    url = `${urlPrefix}/${product.slug}`;
  });

  // Check original content before mocking
  browser
    .drupalRelativeURL(url)
    .waitForElementVisible(page.component)
    .assert.textEquals(page.title, titleFromApi);

  // Modifying the mocked response
  const storedResponse = await getResponseFromLatestByDataType(
    browser,
    dataType,
  );
  const mockedResponseData = JSON.parse(JSON.stringify(storedResponse.value));
  mockedResponseData.data.products.results[0].masterData.current.name =
    titleMocked;

  browser.thSetRequestResponseByHash(
    storedResponse.hash,
    JSON.stringify(mockedResponseData),
  );

  // Activate Mock Mode
  setMockMode(browser);

  // Check if the mocked content is displayed after cache clear
  browser
    .drupalRelativeURL(url)
    .waitForElementVisible(page.component)
    .perform(() => {
      browser.assert.textEquals(page.title, titleFromApi);
    })

    // Checking if the GraphQL response is properly cached.
    .drupalRelativeURL(`${url}?foo=bar`)
    .waitForElementVisible(page.component)
    .perform(() => {
      browser.assert.textEquals(page.title, titleFromApi);
    })

    // Simulate the Commercetools message to invalidate the current product.
    .thDrupalFetchURL(
      `/commercetools-test/message-invalidate-product/${product.id}`,
    )
    .drupalRelativeURL(url)
    .waitForElementVisible(page.component)
    .perform(() => {
      browser.assert.textEquals(page.title, titleMocked);
    })
    .thSetRequestResponseByHash(
      storedResponse.hash,
      JSON.stringify(storedResponse.value),
    )

    // Login as admin to check if the page is cached well for logged in users.
    .thLogin('admin')
    // Check that the mocked title is still on the page despite of
    // the restored stored response, it should be from cache.
    .drupalRelativeURL(url)
    .perform(() => {
      browser
        .waitForElementVisible(page.component)
        .assert.textEquals(page.title, titleMocked);
    })

    // Test the cache clearing functionality.
    .perform(async () => {
      // Set new mocked value.
      mockedResponseData.data.products.results[0].masterData.current.name =
        titleMocked2;
      await browser.thSetRequestResponseByHash(
        storedResponse.hash,
        JSON.stringify(mockedResponseData),
      );
    })
    // Log in as admin and clear the cache on the Commercetools config page.
    .drupalRelativeURL('/admin/config/system/commercetools')
    .click('[data-drupal-selector="edit-advanced-settings"]')
    .click('input[data-drupal-selector="edit-cache-clear"]')
    .perform(async () => {
      browser.assert.textContains(
        messages,
        'commercetools caches has been cleared.',
      );
    })
    .thLogout()

    // Check that the original title is changed to mocked.
    .drupalRelativeURL(url)
    .waitForElementVisible(page.component)
    .perform(() => {
      browser.assert.textEquals(page.title, titleMocked2);
      // Set initial value.
    })
    .thSetRequestResponseByHash(
      storedResponse.hash,
      JSON.stringify(storedResponse.value),
    );

  unsetMockMode(browser);
};
