<?php

namespace Drupal\compare_role_permissions\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\user\Entity\Role;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form to compare permissions between roles.
 */
class CompareRolePermissionsForm extends FormBase {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a new CompareRolePermissionsForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, RendererInterface $renderer) {
    $this->entityTypeManager = $entity_type_manager;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'compare_role_permissions_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#attached']['library'][] = 'compare_role_permissions/compare_role_permissions';

    $form['description'] = [
      '#markup' => '<p>' . $this->t('The compare role permissions module provides two dropdowns. Choose the roles that you want to compare the permissions.') . '</p>',
    ];

    $roles = $this->getRoleOptions();

    $form['role1'] = [
      '#type' => 'select',
      '#title' => $this->t('First Role'),
      '#required' => TRUE,
      '#options' => $roles,
      '#default_value' => $form_state->getValue('role1', 'anonymous'),
    ];

    $form['role2'] = [
      '#type' => 'select',
      '#title' => $this->t('Second Role'),
      '#required' => TRUE,
      '#options' => $roles,
      '#default_value' => $form_state->getValue('role2', 'authenticated'),
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Compare'),
    ];

    // Display results if form has been submitted.
    if ($form_state->get('results_table')) {
      $form['results'] = [
        '#type' => 'details',
        '#title' => $this->t('Comparison Results'),
        '#open' => TRUE,
        'table' => $form_state->get('results_table'),
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $role1 = $form_state->getValue('role1');
    $role2 = $form_state->getValue('role2');

    if ($role1 === $role2) {
      $form_state->setErrorByName('role2', $this->t('Roles should be different from each other.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $role1_id = $form_state->getValue('role1');
    $role2_id = $form_state->getValue('role2');

    $roles = $this->getRoleOptions();
    $role1_name = $roles[$role1_id];
    $role2_name = $roles[$role2_id];

    $header = [
      ['data' => $this->t('Module')],
      ['data' => $this->t('Permission')],
      ['data' => $role1_name],
      ['data' => $role2_name],
    ];

    $role1_permissions = $this->getRolePermissions($role1_id);
    $role2_permissions = $this->getRolePermissions($role2_id);

    $all_permissions = array_merge($role1_permissions, $role2_permissions);
    $different_permissions = array_keys(array_diff_key($role1_permissions, $role2_permissions) +
      array_diff_key($role2_permissions, $role1_permissions));

    $rows = [];
    if (!empty($different_permissions)) {
      foreach ($different_permissions as $permission) {
        $permission_info = $this->getPermissionInfo($permission);
        $module_name = $permission_info['provider'] ?? 'unknown';

        // Create link to permissions page.
        $permissions_url = Url::fromRoute('user.admin_permissions', [], [
          'fragment' => 'module-' . $module_name,
          'attributes' => ['target' => '_blank'],
        ]);
        $module_link = Link::fromTextAndUrl($module_name, $permissions_url);

        $row = [
          'module' => $module_link,
          'permission' => $permission_info['title'] ?? $permission,
          'role1' => isset($role1_permissions[$permission]) ? $this->t('Yes') : $this->t('No'),
          'role2' => isset($role2_permissions[$permission]) ? $this->t('Yes') : $this->t('No'),
        ];
        $rows[] = $row;
      }

      $table = [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No differences found between the selected roles.'),
        '#attributes' => ['class' => ['compare-role-permissions-table']],
      ];

      $form_state->set('results_table', $table);
    }
    else {
      $message = [
        '#markup' => '<p>' . $this->t('Both the selected roles have similar permissions.') . '</p>',
      ];
      $form_state->set('results_table', $message);
    }

    $form_state->setRebuild(TRUE);
  }

  /**
   * Gets role options for the select elements.
   *
   * @return array
   *   An array of role options.
   */
  protected function getRoleOptions() {
    $roles = Role::loadMultiple();
    $options = [];

    foreach ($roles as $role_id => $role) {
      $options[$role_id] = $role->label();
    }

    return $options;
  }

  /**
   * Gets permissions for a specific role.
   *
   * @param string $role_id
   *   The role ID.
   *
   * @return array
   *   An array of permissions keyed by permission machine name.
   */
  protected function getRolePermissions($role_id) {
    $role = Role::load($role_id);
    if (!$role) {
      return [];
    }

    $permissions = $role->getPermissions();
    return array_flip($permissions);
  }

  /**
   * Gets information about a specific permission.
   *
   * @param string $permission
   *   The permission machine name.
   *
   * @return array
   *   An array containing permission information.
   */
  protected function getPermissionInfo($permission) {
    $permissions = \Drupal::service('user.permissions')->getPermissions();
    return $permissions[$permission] ?? [
      'title' => $permission,
      'provider' => 'unknown',
    ];
  }

}
