<?php

namespace Drupal\Tests\compare_role_permissions\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\user\Entity\Role;

/**
 * Functional tests for the Compare Role Permissions module.
 *
 * @group compare_role_permissions
 */
class CompareRolePermissionsFunctionalTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['compare_role_permissions', 'user'];

  /**
   * A user with permission to compare role permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create an admin user with the permission to compare role permissions.
    $this->adminUser = $this->drupalCreateUser(['compare role permissions']);
  }

  /**
   * Tests the compare role permissions form access.
   */
  public function testFormAccess() {
    // Test anonymous user cannot access the form.
    $this->drupalGet('/admin/people/permissions/compare');
    $this->assertSession()->statusCodeEquals(403);

    // Test authenticated user without permission cannot access the form.
    $regular_user = $this->drupalCreateUser();
    $this->drupalLogin($regular_user);
    $this->drupalGet('/admin/people/permissions/compare');
    $this->assertSession()->statusCodeEquals(403);

    // Test admin user can access the form.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/permissions/compare');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Compare role permissions');
  }

  /**
   * Tests the compare role permissions form functionality.
   */
  public function testFormFunctionality() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/permissions/compare');

    // Check that the form elements are present.
    $this->assertSession()->selectExists('role1');
    $this->assertSession()->selectExists('role2');
    $this->assertSession()->buttonExists('Compare');

    // Test form validation - same roles.
    $this->submitForm([
      'role1' => 'anonymous',
      'role2' => 'anonymous',
    ], 'Compare');
    $this->assertSession()->pageTextContains('Roles should be different from each other.');

    // Test successful form submission.
    $this->submitForm([
      'role1' => 'anonymous',
      'role2' => 'authenticated',
    ], 'Compare');
    $this->assertSession()->statusCodeEquals(200);

    // Check that results are displayed.
    $this->assertSession()->pageTextContains('Comparison Results');
  }

  /**
   * Tests permission comparison with custom roles.
   */
  public function testCustomRoleComparison() {
    // Create custom roles with different permissions.
    $role1 = Role::create([
      'id' => 'test_role_1',
      'label' => 'Test Role 1',
    ]);
    $role1->grantPermission('access content');
    $role1->save();

    $role2 = Role::create([
      'id' => 'test_role_2',
      'label' => 'Test Role 2',
    ]);
    $role2->grantPermission('access content');
    $role2->grantPermission('access user profiles');
    $role2->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/permissions/compare');

    // Test comparison of custom roles.
    $this->submitForm([
      'role1' => 'test_role_1',
      'role2' => 'test_role_2',
    ], 'Compare');

    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Comparison Results');

    // Check that the permission difference is shown.
    $this->assertSession()->pageTextContains('access user profiles');
  }

  /**
   * Tests the help text display.
   */
  public function testHelpText() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/permissions/compare');

    // Check that the help text is displayed.
    $this->assertSession()->pageTextContains('Choose the roles that you want to compare');
  }

  /**
   * Tests form behavior when roles have identical permissions.
   */
  public function testIdenticalPermissions() {
    // Create two roles with identical permissions.
    $role1 = Role::create([
      'id' => 'identical_role_1',
      'label' => 'Identical Role 1',
    ]);
    $role1->grantPermission('access content');
    $role1->save();

    $role2 = Role::create([
      'id' => 'identical_role_2',
      'label' => 'Identical Role 2',
    ]);
    $role2->grantPermission('access content');
    $role2->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/people/permissions/compare');

    $this->submitForm([
      'role1' => 'identical_role_1',
      'role2' => 'identical_role_2',
    ], 'Compare');

    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Both the selected roles have similar permissions');
  }

}
