<?php

namespace Drupal\Tests\compare_role_permissions\Unit;

use Drupal\compare_role_permissions\Form\CompareRolePermissionsForm;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\user\Entity\Role;

/**
 * Unit tests for the CompareRolePermissionsForm class.
 *
 * @group compare_role_permissions
 * @coversDefaultClass \Drupal\compare_role_permissions\Form\CompareRolePermissionsForm
 */
class CompareRolePermissionsFormTest extends UnitTestCase {

  /**
   * The form under test.
   *
   * @var \Drupal\compare_role_permissions\Form\CompareRolePermissionsForm
   */
  protected $form;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * Mock renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $renderer;

  /**
   * Mock form state.
   *
   * @var \Drupal\Core\Form\FormStateInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $formState;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->renderer = $this->createMock(RendererInterface::class);
    $this->formState = $this->createMock(FormStateInterface::class);

    $this->form = new CompareRolePermissionsForm(
      $this->entityTypeManager,
      $this->renderer
    );
  }

  /**
   * Tests the form ID.
   *
   * @covers ::getFormId
   */
  public function testGetFormId() {
    $this->assertEquals('compare_role_permissions_form', $this->form->getFormId());
  }

  /**
   * Tests form validation with same roles.
   *
   * @covers ::validateForm
   */
  public function testValidateFormSameRoles() {
    $form = [];

    $this->formState->expects($this->exactly(2))
      ->method('getValue')
      ->willReturnMap([
        ['role1', 'admin'],
        ['role2', 'admin'],
      ]);

    $this->formState->expects($this->once())
      ->method('setErrorByName')
      ->with('role2', $this->anything());

    $this->form->validateForm($form, $this->formState);
  }

  /**
   * Tests form validation with different roles.
   *
   * @covers ::validateForm
   */
  public function testValidateFormDifferentRoles() {
    $form = [];

    $this->formState->expects($this->exactly(2))
      ->method('getValue')
      ->willReturnMap([
        ['role1', 'admin'],
        ['role2', 'editor'],
      ]);

    $this->formState->expects($this->never())
      ->method('setErrorByName');

    $this->form->validateForm($form, $this->formState);
  }

  /**
   * Tests the buildForm method structure.
   *
   * @covers ::buildForm
   */
  public function testBuildFormStructure() {
    $form_array = [];
    $form_state = $this->createMock(FormStateInterface::class);

    $form_state->method('getValue')->willReturnMap([
      ['role1', 'anonymous'],
      ['role2', 'authenticated'],
    ]);

    $form_state->method('get')->with('results_table')->willReturn(NULL);

    $result = $this->form->buildForm($form_array, $form_state);

    // Test that required form elements exist.
    $this->assertArrayHasKey('role1', $result);
    $this->assertArrayHasKey('role2', $result);
    $this->assertArrayHasKey('submit', $result);
    $this->assertArrayHasKey('description', $result);

    // Test form element types.
    $this->assertEquals('select', $result['role1']['#type']);
    $this->assertEquals('select', $result['role2']['#type']);
    $this->assertEquals('submit', $result['submit']['#type']);

    // Test that form elements are required.
    $this->assertTrue($result['role1']['#required']);
    $this->assertTrue($result['role2']['#required']);
  }

}
