<?php

declare(strict_types=1);

namespace Drupal\compiler_scss;

use Drupal\compiler\Plugin\CompilerPluginInterface;

/**
 * An interface to facilitate altering the SCSS compiler plugin.
 *
 * Copyright (C) 2025  Library Solutions, LLC (et al.).
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
interface CompilerInterface extends CompilerPluginInterface {

  /**
   * Set a user-defined host function to make available to the compiler.
   *
   * @param string $identifier
   *   The identifier to use for the function.
   * @param \Closure(list<\ScssPhp\ScssPhp\Value\Value>):\ScssPhp\ScssPhp\Value\Value $callback
   *   The host callback function.
   * @param list<string> $arguments
   *   A list of argument declarations for the host function (default: []).
   * @param bool $overwrite
   *   Whether or not to permit overwriting an existing value (default: FALSE).
   *
   * @throws \InvalidArgumentException
   *   If the callback or identifier cannot be used.
   * @throws \RuntimeException
   *   If $overwrite is TRUE, and the supplied identifier was already set.
   *
   * @see ::processFunctionArguments()
   *   For more information about the format of each argument declaration.
   *
   * @return string
   *   The actual identifier of the function.
   */
  public function setFunction(string $identifier, \Closure $callback, array $arguments = [], bool $overwrite = FALSE): string;

  /**
   * Set a variable to inject into the compiler.
   *
   * The supplied identifier will be sanitized before the variable is set.
   *
   * The supplied value will be recursively converted to compiler-compatible
   * types as needed. Values that are already compatible with the compiler will
   * not be modified by this method.
   *
   * @param string $identifier
   *   The identifier to use for the variable.
   * @param mixed $value
   *   The value of the variable.
   * @param bool $overwrite
   *   Whether or not to permit overwriting an existing value (default: FALSE).
   *
   * @throws \InvalidArgumentException
   *   If the identifier cannot be used.
   * @throws \RuntimeException
   *   If $overwrite is TRUE, and the supplied identifier was already set.
   *
   * @return string
   *   The actual identifier of the variable.
   */
  public function setVariable(string $identifier, mixed $value, bool $overwrite = FALSE): string;

  /**
   * Unset the function with the supplied identifier.
   *
   * This method will always succeed, even if the identifier isn't valid or
   * associated with a function.
   *
   * @param string $identifier
   *   The identifier to unset.
   */
  public function unsetFunction(string $identifier): void;

  /**
   * Unset the variable with the supplied identifier.
   *
   * This method will always succeed, even if the identifier isn't valid or
   * associated with a variable.
   *
   * @param string $identifier
   *   The identifier to unset.
   */
  public function unsetVariable(string $identifier): void;

}
