<?php

namespace Drupal\compiler_scss\Plugin\Compiler;

use Drupal\compiler\CompilerContextInterface;
use Drupal\compiler\CompilerInputDirect;
use Drupal\compiler\CompilerInputFile;
use Drupal\compiler\Plugin\CompilerPluginBase;

use Drupal\compiler_scss\CompilerInterface;
use Drupal\compiler_scss\ValueConverter;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;

use ScssPhp\ScssPhp\Compiler as ScssCompiler;
use ScssPhp\ScssPhp\Value\Value as SassValue;
use ScssPhp\ScssPhp\ValueConverter as SassValueConverter;

use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The SCSS compiler plugin.
 *
 * Copyright (C) 2025  Library Solutions, LLC (et al.).
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * @Compiler("compiler_scss")
 */
final class Compiler extends CompilerPluginBase implements CompilerInterface, ContainerFactoryPluginInterface {

  /**
   * A map of functions to inject into the compiler keyed by identifier.
   *
   * @var array<string,array{0:\Closure(list<\ScssPhp\ScssPhp\Value\Value>):\ScssPhp\ScssPhp\Value\Value,1:list<string>}>
   */
  private array $functions = [];

  /**
   * The value converter service.
   *
   * @var \Drupal\compiler_scss\ValueConverter
   */
  private ValueConverter $valueConverter;

  /**
   * A map of variables to inject into the compiler keyed by identifier.
   *
   * @var array<string,\ScssPhp\ScssPhp\Value\Value>
   */
  private array $variables = [];

  /**
   * Constructs a Compiler object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\compiler_scss\ValueConverter $value_converter
   *   The value converter service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ValueConverter $value_converter) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->valueConverter = $value_converter;
  }

  /**
   * Clean an identifier for use in the compiler.
   *
   * This method applies the following steps to the supplied input:
   *
   * 1. Replace all non-alphanumeric code point spans with a hyphen.
   * 2. Make the resulting string lowercase.
   * 3. Remove all leading non-alpha character spans and trailing hyphen spans.
   *
   * If the supplied identifier cannot be used, then an empty string will be
   * returned by this method.
   *
   * @param string $identifier
   *   A UTF-8 identifier string to clean.
   *
   * @return string
   *   The cleaned identifier, or an empty string.
   */
  private function cleanIdentifier(string $identifier): string {
    $identifier = \preg_replace('/[^0-9A-Za-z]+/u', '-', $identifier);

    if (\is_string($identifier)) {
      $identifier = \strtolower($identifier);
      $identifier = \preg_replace('/^[^a-z]+|-+$/', '', $identifier);

      if (\is_string($identifier)) {
        return $identifier;
      }
    }

    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function compile(CompilerContextInterface $context) {
    $compiler = new ScssCompiler();

    foreach ($this->functions as $identifier => [$callback, $args]) {
      $compiler->registerFunction($identifier, $callback, $args);
    }

    $compiler->addVariables($this->variables);
    $compiler->setImportPaths($context->getOption('import_paths') ?: []);

    $input = $this->getInput($context);

    $result = $compiler->compileString($input);
    $result = $result->getCss();

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('compiler_scss.value_converter'),
    );
  }

  /**
   * Retrieve the source code to pass to the compiler.
   *
   * @param \Drupal\compiler\CompilerContextInterface $context
   *   A compiler context used to define a compilation.
   *
   * @return string
   *   The source code to pass to the compiler.
   */
  private function getInput(CompilerContextInterface $context) {
    $source = [];

    // Iterate through each compiler input to construct the compiler source.
    foreach (new \RecursiveIteratorIterator($context->getInputs()) as $input) {
      $result = $input->get();

      if ($input instanceof CompilerInputFile) {
        $result = \file_get_contents($result);
      }
      elseif (!$input instanceof CompilerInputDirect) {
        throw new \RuntimeException('Unsupported input type');
      }

      $source[] = $result;
    }

    // Concatenate the array of source code into a single string.
    return \implode("\r\n", $source);
  }

  /**
   * Process a list of Sass function argument declarations.
   *
   * This method validates the supplied argument list to ensure that it conforms
   * to a rigid set of standards to reduce the incidence of runtime errors:
   *
   * - All argument names must be non-empty, may only contain alphanumeric
   *   characters and hyphens, must start with an alpha character, and must not
   *   end with a hyphen.
   * - Argument names may optionally be suffixed by "..." to indicate that the
   *   argument is variadic.
   * - No argument declarations may follow a variadic argument.
   * - Default values may be specified by suffixing non-variadic declarations
   *   with a colon, followed by a string representation of a Sass value.
   *
   * @param list<string> $arguments
   *   A list of Sass function argument declarations.
   *
   * @throws \InvalidArgumentException
   *   If the supplied argument list is invalid.
   *
   * @return list<string>
   *   A validated, and canonicalized list of argument declarations.
   */
  private function processFunctionArguments(array $arguments): array {
    $result = [];

    $encountered_variadic_argument = FALSE;

    foreach ($arguments as $declaration) {
      if ($encountered_variadic_argument) {
        throw new \InvalidArgumentException('Additional argument declarations are not allowed to follow a variadic argument declaration');
      }

      // Extract the name and default value of this argument declaration.
      $declaration = \preg_split('/:/u', $declaration, 2);
      \assert($declaration !== FALSE);

      $name = $declaration[0];
      $default_value = $declaration[1] ?? NULL;

      // Validate that the argument name conforms to the expected format.
      if (!\preg_match('/^(?=[A-Za-z])(?:[-0-9A-Za-z]+)(?<!-)(?:\\.\\.\\.)?$/u', $name)) {
        throw new \InvalidArgumentException('One or more argument identifiers are invalid; argument identifiers must be non-empty, may only contain alphanumeric characters and hyphens, must start with an alpha character, and must not end with a hyphen. Argument identifiers may optionally be suffixed by "..." to indicate that the argument is variadic');
      }

      // Set the flag used to track whether a variadic argument was encountered.
      $encountered_variadic_argument = \substr($name, -3) === '...';

      if (isset($default_value)) {
        if ($encountered_variadic_argument) {
          throw new \InvalidArgumentException('The argument declaration ' . \var_export($name, TRUE) . ' is variadic, so it cannot have a default value');
        }

        try {
          // Attempt to validate and canonicalize the specified default value.
          $default_value = SassValueConverter::parseValue($default_value);
          $default_value = (string) $default_value;
        }
        catch (\Throwable) {
          throw new \InvalidArgumentException("The default value for identifier '{$name}' is invalid");
        }
      }

      // Store the canonicalized argument declaration.
      $result[] = isset($default_value) ? "{$name}:{$default_value}" : $name;
    }

    return $result;
  }

  /**
   * Set a user-defined host function to make available to the compiler.
   *
   * The host callback function may accept no parameters, or exactly one
   * parameter containing a list of Sass values. The return value will be
   * implicitly converted to a Sass value.
   *
   * @param string $identifier
   *   The identifier to use for the function.
   * @param \Closure(list<\ScssPhp\ScssPhp\Value\Value>|void):mixed $callback
   *   The host callback function.
   * @param list<string> $arguments
   *   A list of argument declarations for the function (default: []).
   * @param bool $overwrite
   *   Whether or not to permit overwriting an existing value (default: FALSE).
   *
   * @throws \InvalidArgumentException
   *   If the callback or identifier cannot be used.
   * @throws \RuntimeException
   *   If $overwrite is TRUE, and the supplied identifier was already set.
   *
   * @see ::processFunctionArguments()
   *   For more information about the format of each argument declaration.
   *
   * @return string
   *   The actual identifier of the function.
   */
  public function setFunction(string $identifier, \Closure $callback, array $arguments = [], bool $overwrite = FALSE): string {
    if (!$identifier = $this->cleanIdentifier($identifier)) {
      throw new \InvalidArgumentException('The supplied identifier cannot be used');
    }

    if (!$overwrite && \array_key_exists($identifier, $this->functions)) {
      throw new \RuntimeException("A function with identifier '{$identifier}' already exists");
    }

    $reflection = new \ReflectionFunction($callback);
    $parameters = $reflection->getParameters();

    $first_parameter = $parameters[0] ?? NULL;
    $first_parameter_type = $first_parameter?->getType();

    if (\count($parameters) > 1 || ($first_parameter && (!$first_parameter_type instanceof \ReflectionNamedType || $first_parameter_type->getName() !== 'array'))) {
      throw new \InvalidArgumentException('The supplied callback function may accept no parameters, or exactly one parameter containing a list of Sass values');
    }

    $callback = fn (array $args): SassValue => $this->valueConverter->convertValueRecursive($callback($args));
    $this->functions[$identifier] = [$callback, $this->processFunctionArguments($arguments)];

    return $identifier;
  }

  /**
   * Set a variable to inject into the compiler.
   *
   * The supplied identifier will be sanitized before the variable is set.
   *
   * The supplied value will be recursively converted to compiler-compatible
   * types as needed. Values that are already compatible with the compiler will
   * not be modified by this method.
   *
   * @param string $identifier
   *   The identifier to use for the variable.
   * @param mixed $value
   *   The value of the variable.
   * @param bool $overwrite
   *   Whether or not to permit overwriting an existing value (default: FALSE).
   *
   * @throws \InvalidArgumentException
   *   If the identifier cannot be used.
   * @throws \RuntimeException
   *   If $overwrite is TRUE, and the supplied identifier was already set.
   *
   * @return string
   *   The actual identifier of the variable.
   */
  public function setVariable(string $identifier, mixed $value, bool $overwrite = FALSE): string {
    if (!$identifier = $this->cleanIdentifier($identifier)) {
      throw new \InvalidArgumentException('The supplied identifier cannot be used');
    }

    if (!$overwrite && \array_key_exists($identifier, $this->variables)) {
      throw new \RuntimeException("A variable with identifier '{$identifier}' already exists");
    }

    $this->variables[$identifier] = $this->valueConverter->convertValueRecursive($value);

    return $identifier;
  }

  /**
   * Unset the function with the supplied identifier.
   *
   * This method will always succeed, even if the identifier isn't valid or
   * associated with a function.
   *
   * @param string $identifier
   *   The identifier to unset.
   */
  public function unsetFunction(string $identifier): void {
    if ($identifier = $this->cleanIdentifier($identifier)) {
      unset($this->functions[$identifier]);
    }
  }

  /**
   * Unset the variable with the supplied identifier.
   *
   * This method will always succeed, even if the identifier isn't valid or
   * associated with a variable.
   *
   * @param string $identifier
   *   The identifier to unset.
   */
  public function unsetVariable(string $identifier): void {
    if ($identifier = $this->cleanIdentifier($identifier)) {
      unset($this->variables[$identifier]);
    }
  }

}
