<?php

declare(strict_types=1);

namespace Drupal\compiler_scss;

use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * An enumeration of units supported by this module.
 *
 * Copyright (C) 2025  Library Solutions, LLC (et al.).
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
enum Unit: string {

  // Absolute units.
  case Centimeters = 'cm';
  case Inches = 'in';
  case Millimeters = 'mm';
  case Picas = 'pc';
  case Pixels = 'px';
  case Points = 'pt';

  // Angle units.
  case Degrees = 'deg';
  case Gradians = 'grad';
  case Radians = 'rad';
  case Turns = 'turn';

  // Frequency units.
  case Hertz = 'Hz';
  case Kilohertz = 'kHz';

  // Percent unit.
  case Percent = '%';

  // Relative units.
  case ElementFontHeightLowercase = 'ex';
  case ElementFontSize = 'em';
  case LineHeight = 'lh';
  case RootElementFontSize = 'rem';
  case ViewportHeight = 'vh';
  case ViewportMaximum = 'vmax';
  case ViewportMinimum = 'vmin';
  case ViewportWidth = 'vw';
  case ZeroCharacterWidth = 'ch';

  // Resolution units.
  case DotsPerCentimeter = 'dpcm';
  case DotsPerInch = 'dpi';
  case DotsPerPixel = 'dppx';

  // Time units.
  case Milliseconds = 'ms';
  case Seconds = 's';

  /**
   * Get the unit's label.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The unit's label.
   */
  public function label(): TranslatableMarkup {
    $options['context'] = 'Sass unit label';

    return match ($this) {
      // Absolute units.
      self::Centimeters => \t('centimeters', options: $options),
      self::Inches => \t('inches', options: $options),
      self::Millimeters => \t('millimeters', options: $options),
      self::Picas => \t('picas', options: $options),
      self::Pixels => \t('pixels', options: $options),
      self::Points => \t('points', options: $options),

      // Angle units.
      self::Degrees => \t('degrees', options: $options),
      self::Gradians => \t('gradians', options: $options),
      self::Radians => \t('radians', options: $options),
      self::Turns => \t('turns', options: $options),

      // Frequency units.
      self::Hertz => \t('hertz', options: $options),
      self::Kilohertz => \t('kilohertz', options: $options),

      // Percent unit.
      self::Percent => \t('percent', options: $options),

      // Relative units.
      self::ElementFontHeightLowercase => \t('&times; the height of lowercase characters in the element font', options: $options),
      self::ElementFontSize => \t('&times; the element font size', options: $options),
      self::LineHeight => \t('&times; the line height of the element', options: $options),
      self::RootElementFontSize => \t('&times; the root element font size', options: $options),
      self::ViewportHeight => \t('&times; 1% of the viewport height', options: $options),
      self::ViewportMaximum => \t('&times; 1% of the larger viewport dimension', options: $options),
      self::ViewportMinimum => \t('&times; 1% of the smaller viewport dimension', options: $options),
      self::ViewportWidth => \t('&times; 1% of the viewport width', options: $options),
      self::ZeroCharacterWidth => \t('&times; the width of "0" in the element font', options: $options),

      // Resolution units.
      self::DotsPerCentimeter => \t('dots per centimeter', options: $options),
      self::DotsPerInch => \t('dots per inch', options: $options),
      self::DotsPerPixel => \t('dots per pixel', options: $options),

      // Time units.
      self::Milliseconds => \t('milliseconds', options: $options),
      self::Seconds => \t('seconds', options: $options),
    };
  }

}
