<?php

declare(strict_types=1);

namespace Drupal\compiler_scss;

/**
 * An enumeration of unit groups.
 *
 * Copyright (C) 2025  Library Solutions, LLC (et al.).
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */
enum UnitGroup {

  case Absolute;
  case Angle;
  case Frequency;
  case Length;
  case LengthPercent;
  case Percent;
  case Relative;
  case Resolution;
  case Time;

  /**
   * Get a list of unit options.
   *
   * @param list<\Drupal\compiler_scss\Unit> $units
   *   A list of units.
   *
   * @return array<string,\Drupal\Core\StringTranslation\TranslatableMarkup>
   *   A list of unit options.
   */
  public static function options(array $units): array {
    $options = [];

    foreach ($units as $unit) {
      $options[$unit->value] = $unit->label();
    }

    return $options;
  }

  /**
   * Get the group's units.
   *
   * @return list<\Drupal\compiler_scss\Unit>
   *   The group's units.
   */
  public function units(): array {
    return match ($this) {
      self::Absolute => [
        Unit::Centimeters,
        Unit::Inches,
        Unit::Millimeters,
        Unit::Picas,
        Unit::Pixels,
        Unit::Points,
      ],
      self::Angle => [
        Unit::Degrees,
        Unit::Gradians,
        Unit::Radians,
        Unit::Turns,
      ],
      self::Frequency => [
        Unit::Hertz,
        Unit::Kilohertz,
      ],
      self::Length => [
        Unit::Centimeters,
        Unit::Inches,
        Unit::Millimeters,
        Unit::Picas,
        Unit::Pixels,
        Unit::Points,
        Unit::ElementFontHeightLowercase,
        Unit::ElementFontSize,
        Unit::LineHeight,
        Unit::RootElementFontSize,
        Unit::ViewportHeight,
        Unit::ViewportMaximum,
        Unit::ViewportMinimum,
        Unit::ViewportWidth,
        Unit::ZeroCharacterWidth,
      ],
      self::LengthPercent => [
        Unit::Centimeters,
        Unit::Inches,
        Unit::Millimeters,
        Unit::Picas,
        Unit::Pixels,
        Unit::Points,
        Unit::Percent,
        Unit::ElementFontHeightLowercase,
        Unit::ElementFontSize,
        Unit::LineHeight,
        Unit::RootElementFontSize,
        Unit::ViewportHeight,
        Unit::ViewportMaximum,
        Unit::ViewportMinimum,
        Unit::ViewportWidth,
        Unit::ZeroCharacterWidth,
      ],
      self::Percent => [
        Unit::Percent,
      ],
      self::Relative => [
        Unit::ElementFontHeightLowercase,
        Unit::ElementFontSize,
        Unit::LineHeight,
        Unit::RootElementFontSize,
        Unit::ViewportHeight,
        Unit::ViewportMaximum,
        Unit::ViewportMinimum,
        Unit::ViewportWidth,
        Unit::ZeroCharacterWidth,
      ],
      self::Resolution => [
        Unit::DotsPerCentimeter,
        Unit::DotsPerInch,
        Unit::DotsPerPixel,
      ],
      self::Time => [
        Unit::Milliseconds,
        Unit::Seconds,
      ],
    };
  }

}
