<?php

declare(strict_types=1);

namespace Drupal\compiler_scss;

use ScssPhp\ScssPhp\Collection\Map;
use ScssPhp\ScssPhp\Value\ListSeparator;
use ScssPhp\ScssPhp\Value\SassBoolean;
use ScssPhp\ScssPhp\Value\SassColor;
use ScssPhp\ScssPhp\Value\SassList;
use ScssPhp\ScssPhp\Value\SassMap;
use ScssPhp\ScssPhp\Value\SassNull;
use ScssPhp\ScssPhp\Value\SassString;
use ScssPhp\ScssPhp\Value\SingleUnitSassNumber;
use ScssPhp\ScssPhp\Value\UnitlessSassNumber;
use ScssPhp\ScssPhp\Value\Value;

/**
 * An implicit value converter from native value types to Sass value types.
 *
 * Copyright (C) 2025  Library Solutions, LLC (et al.).
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * @internal
 */
final class ValueConverter {

  /**
   * Attempt to coerce the supplied value to a Sass color.
   *
   * If the supplied value is an array with the following structure, then it
   * will be coerced to a Sass color:
   *
   * - red (integer from 0 to 255)
   * - green (integer from 0 to 255)
   * - blue (integer from 0 to 255)
   * - alpha (float from 0 to 1; optional)
   *
   * Otherwise, the supplied value is returned without modification.
   *
   * @param mixed $input
   *   The value to possibly coerce to a Sass color.
   *
   * @return mixed
   *   The coerced or supplied value.
   */
  private function coerceValueToColor(mixed $input): mixed {
    $shape = ['red' => TRUE, 'green' => TRUE, 'blue' => TRUE, 'alpha' => FALSE];

    if ($array = $this->getArrayWithShape($input, $shape)) {
      $red = $this->getInteger($array['red'], 0, 255);
      $green = $this->getInteger($array['green'], 0, 255);
      $blue = $this->getInteger($array['blue'], 0, 255);
      $alpha = $this->getFloat(@$array['alpha'] ?? 1.0, 0.0, 1.0);

      if (isset($red, $green, $blue, $alpha)) {
        return SassColor::rgb($red, $green, $blue, $alpha);
      }
    }

    return $input;
  }

  /**
   * Attempt to coerce the supplied value to a single-unit Sass number.
   *
   * If the supplied value is an array with the following structure, then it
   * will be coerced to a Sass number:
   *
   * - number (float or integer)
   * - unit (string)
   *
   * Otherwise, the supplied value is returned without modification.
   *
   * @param mixed $input
   *   The value to possibly coerce to a Sass number.
   *
   * @return mixed
   *   The coerced or supplied value.
   */
  private function coerceValueToSingleUnitNumber(mixed $input): mixed {
    $shape = ['value' => TRUE, 'unit' => TRUE];

    if ($array = $this->getArrayWithShape($input, $shape)) {
      $value = $this->getFloat($array['value']);
      $unit = $this->getString($array['unit']);

      if (isset($value, $unit)) {
        return new SingleUnitSassNumber($value, $unit);
      }
    }

    return $input;
  }

  /**
   * Convert an array into a Sass map.
   *
   * @param array<\ScssPhp\ScssPhp\Value\Value> $value
   *   The array to convert.
   *
   * @return \ScssPhp\ScssPhp\Value\SassMap
   *   The converted value.
   */
  private function convertArrayToSassMap(array $value): SassMap {
    /** @var \ScssPhp\ScssPhp\Collection\Map<\ScssPhp\ScssPhp\Value\Value> */
    $map = new Map();

    foreach ($value as $key => $value) {
      $map->put($this->convertValue($key), $value);
    }

    return SassMap::create($map);
  }

  /**
   * Convert a plain data object into an array.
   *
   * If the object is not a plain data object, then it will not be modified.
   *
   * This method is not recursive.
   *
   * @param object $object
   *   The object to convert.
   *
   * @return array<mixed>|object
   *   The converted or original object.
   */
  private function convertObjectToArray(object $object): array|object {
    if (!\is_subclass_of($object, \stdClass::class, FALSE) && $object instanceof \stdClass) {
      $object = \get_object_vars($object);
    }

    return $object;
  }

  /**
   * Convert plain data objects into arrays, recursively.
   *
   * @param object $object
   *   The object to convert.
   *
   * @return array<mixed>|object
   *   The converted or original object.
   */
  private function convertObjectToArrayRecursive(object $object): array|object {
    $result = $this->convertObjectToArray($object);

    if (\is_array($result)) {
      \array_walk_recursive($result, function (mixed &$element): void {
        if (\is_object($element)) {
          $element = $this->convertObjectToArrayRecursive($element);
        }
      });
    }

    return $result;
  }

  /**
   * Convert the supplied value into a Sass value.
   *
   * @param mixed $value
   *   The value to convert.
   *
   * @throws \UnexpectedValueException
   *   If the value cannot be implicitly converted.
   *
   * @return \ScssPhp\ScssPhp\Value\Value
   *   A Sass value.
   */
  private function convertValue(mixed $value): Value {
    if (\is_null($value)) {
      $value = SassNull::create();
    }

    if (\is_bool($value)) {
      $value = SassBoolean::create($value);
    }

    if (\is_float($value) || \is_int($value)) {
      $value = new UnitlessSassNumber($value);
    }

    if (\is_string($value)) {
      $value = new SassString($value);
    }

    if (!$value instanceof Value) {
      throw new \UnexpectedValueException('There is no implicit conversion defined for a value of type ' . $this->getReadableType($value));
    }

    return $value;
  }

  /**
   * Convert the supplied value into a Sass value.
   *
   * This method applies the following transformations to the supplied value:
   *
   * 1. If the value is a plain data object, then it will be transformed into a
   *    nested array structure.
   * 2. If the value is an array that can be coerced, then it will be coerced
   *    directly into the appropriate type.
   * 3. If the value is an array, then this method will be recursively invoked
   *    for each element of the array, and the array will then be transformed
   *    into the appropriate data structure (either a Sass list or map).
   * 4. The value will be transformed as an atomic type.
   *
   * The last step will be skipped for values that have already been transformed
   * into Sass values, and an exception will be thrown for types without an
   * implicit conversion pathway.
   *
   * This method implicitly converts all lists into comma-separated, unbracketed
   * Sass lists. Use `list.join()` to change a list's configuration:
   *
   * ```scss
   * $output: list.join($input, (), $separator: space, $bracketed: true);
   * ```
   *
   * @param mixed $value
   *   The value to convert.
   *
   * @throws \UnexpectedValueException
   *   If a value could not be converted into a Sass value.
   *
   * @see ::coerceValueToColor()
   *   For more information about which arrays will be coerced to a color.
   * @see https://sass-lang.com/documentation/modules/list/#join
   *   Documentation for the `list.join()` built-in function.
   *
   * @return \ScssPhp\ScssPhp\Value\Value
   *   The converted value.
   */
  public function convertValueRecursive(mixed $value): Value {
    if (\is_object($value)) {
      $value = $this->convertObjectToArrayRecursive($value);
    }

    $value = $this->coerceValueToColor($value);
    $value = $this->coerceValueToSingleUnitNumber($value);

    if (\is_array($value)) {
      $value = \array_map($this->convertValueRecursive(...), $value);
      $value = \array_is_list($value) ? new SassList($value, ListSeparator::COMMA) : $this->convertArrayToSassMap($value);
    }

    $value = $this->convertValue($value);

    return $value;
  }

  /**
   * Get an array of the specified shape from the supplied value.
   *
   * @param mixed $value
   *   The value to check.
   * @param bool[] $shape
   *   An array whose values indicate whether their key is required.
   *
   * @return array<mixed>|null
   *   An array of the specified shape.
   */
  private function getArrayWithShape(mixed $value, array $shape): array|null {
    $shape_required = \array_filter($shape);

    if (\is_array($value) && !\array_diff_key($value, $shape) && !\array_diff_key($shape_required, $value)) {
      return $value;
    }

    return NULL;
  }

  /**
   * Get a float from the supplied value.
   *
   * @param mixed $value
   *   The value to check.
   * @param float|null $min
   *   The minimum allowed value, or NULL if none.
   * @param float|null $max
   *   The maximum allowed value, or NULL if none.
   *
   * @return float|null
   *   A float, or NULL.
   */
  private function getFloat(mixed $value, ?float $min = NULL, ?float $max = NULL): float|null {
    $value = \filter_var($value, \FILTER_VALIDATE_FLOAT, $this->getNumberFilterOptions($min, $max));

    if ($value !== FALSE) {
      return $value;
    }

    return NULL;
  }

  /**
   * Get an integer from the supplied value.
   *
   * @param mixed $value
   *   The value to check.
   * @param int|null $min
   *   The minimum allowed value, or NULL if none.
   * @param int|null $max
   *   The maximum allowed value, or NULL if none.
   *
   * @return int|null
   *   An integer, or NULL.
   */
  private function getInteger(mixed $value, ?int $min = NULL, ?int $max = NULL): int|null {
    $value = \filter_var($value, \FILTER_VALIDATE_INT, $this->getNumberFilterOptions($min, $max));

    if ($value !== FALSE) {
      return $value;
    }

    return NULL;
  }

  /**
   * Get number filter options.
   *
   * @param float|int|null $min
   *   The minimum allowed value, or NULL if none.
   * @param float|int|null $max
   *   The maximum allowed value, or NULL if none.
   *
   * @return array<mixed>
   *   The number filter options.
   */
  private function getNumberFilterOptions(float|int|null $min = NULL, float|int|null $max = NULL): array {
    $options = [];

    if (isset($min)) {
      $options['min_range'] = $min;
    }

    if (isset($max)) {
      $options['max_range'] = $max;
    }

    return $options;
  }

  /**
   * Get a human-readable type for the supplied value.
   *
   * @param mixed $value
   *   The value for which to get its type.
   *
   * @return string
   *   A human-readable type for the supplied value.
   */
  private function getReadableType(mixed $value): string {
    $type = \gettype($value);

    if (\is_object($value)) {
      $type = \get_class($value);
    }

    if (\is_resource($value)) {
      $type = \get_resource_type($value);
      $type = "resource ({$type})";
    }

    return $type;
  }

  /**
   * Get a string from the supplied value.
   *
   * @param mixed $value
   *   The value to check.
   *
   * @return string|null
   *   A string, or NULL.
   */
  private function getString(mixed $value): string|null {
    $value = \filter_var($value, \FILTER_DEFAULT);

    if ($value !== FALSE) {
      return $value;
    }

    return NULL;
  }

}
