<?php

namespace Drupal\complete_webform_exporter\Controller;

use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Drupal\complete_webform_exporter\Service\ExporterService;
use Drupal\Core\File\FileSystemInterface;
use Drupal\webform\Entity\Webform;
use Drupal\webform\Entity\WebformSubmission;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Drupal\Core\Controller\ControllerBase;

/**
 * Returns responses for Webform Each Submission Data Download routes.
 */
class CompleteWebFormExporterController extends ControllerBase {

  public function __construct(
    protected readonly FileSystemInterface $fileSystem,
    protected readonly ExporterService $exporterService,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('file_system'),
      $container->get('complete_webform_exporter.exporter')
    );
  }

  /**
   * Downloads a ZIP file with the submission in an Excel and attached file.
   */
  public function downloadDownload($webform_id, $submission_id): BinaryFileResponse {
    $webform = Webform::load($webform_id);
    $submission = WebformSubmission::load($submission_id);
    if (!$webform || !$submission) {
      throw new NotFoundHttpException();
    }

    // Build headers and one row.
    $elements = $webform->getElementsDecodedAndFlattened();
    $headers = $this->exporterService->buildHeaders($elements);
    $row = $this->exporterService->buildRow($webform, $submission, $elements);

    // Create excel.
    $excel_uri = $this->exporterService->createExcel($headers, [$row], $webform_id . '-' . $submission_id);

    // Collect fids and signatures real paths.
    $fids = $this->exporterService->collectFileIds($webform, $submission);

    // Gather signature real paths (scan elements to find signature elements).
    $signature_paths = [];
    foreach ($elements as $key => $element) {
      if (isset($element['#type']) && $element['#type'] === 'webform_signature' && array_key_exists($key, $submission->getData())) {
        $path = $this->exporterService->getSignatureRealpath($webform, $submission, $element, $submission->getData()[$key]);
        if ($path) {
          $signature_paths[] = $path;
        }
      }
    }

    // Build zip (adds attachments and Excel and signatures).
    $base_name = "{$webform_id}-{$submission_id}-files";
    $zip_path = $this->exporterService->createZip($base_name, function ($archive) use ($fids, $signature_paths, $excel_uri) {
      // Add fids with no prefix.
      $this->exporterService->addFilesToArchive($archive, $fids);
      // Add signatures.
      $this->exporterService->addRealPathsToArchive($archive, $signature_paths);
      // Add the Excel file.
      $archive->getArchive()->addFile($this->fileSystem->realpath($excel_uri), basename($excel_uri));
    });

    // Remove temporary Excel.
    $this->fileSystem->delete($excel_uri);

    $download_name = basename($zip_path);
    $headers = [
      'Content-Type' => 'application/zip',
      'Content-Disposition' => 'attachment; filename="' . $download_name . '"',
      'Content-Description' => 'File Transfer',
    ];

    return new BinaryFileResponse($zip_path, 200, $headers, TRUE);
  }

}
