<?php

namespace Drupal\complete_webform_exporter\Plugin\Action;

use Drupal\complete_webform_exporter\Service\ExporterService;
use Drupal\Core\Action\ActionBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\webform\WebformSubmissionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

/**
 * Export the webform submissions.
 *
 * @Action(
 *   id = "webform_submission_export_action",
 *   label = @Translation("Export submission"),
 *   type = "webform_submission"
 * )
 */
class WebformSubmissionsExporterAction extends ActionBase implements ContainerFactoryPluginInterface {

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly FileSystemInterface $fileSystem,
    protected readonly ExporterService $exporterService,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('file_system'),
      $container->get('complete_webform_exporter.exporter')
    );
  }

  /**
   * Executes the bulk action and streams a ZIP file to the browser.
   *
   * @param \Drupal\Core\Entity\EntityInterface[] $entities
   *   The entities to act upon.
   */
  public function executeMultiple(array $entities): void {
    if (empty($entities)) {
      return;
    }

    /** @var \Drupal\webform\WebformSubmissionInterface $first */
    $first = reset($entities);
    if (!$first instanceof WebformSubmissionInterface) {
      return;
    }

    $webform_id = $first->getWebform()->id();
    $timestamp = date('Y-m-d-H-i-s');
    $zip_basename = $webform_id . '-submissions-' . $timestamp;

    // Create a single Excel containing all selected submissions and a zip.
    $elements = $first->getWebform()->getElementsDecodedAndFlattened();
    $headers = $this->exporterService->buildHeaders($elements);

    // Build rows for all submissions.
    $rows = [];
    $all_fids = [];
    $signature_paths = [];

    foreach ($entities as $entity) {
      if (!$entity instanceof WebformSubmissionInterface) {
        continue;
      }
      $rows[] = $this->exporterService->buildRow($entity->getWebform(), $entity, $elements);

      // Collect file ids for this submission.
      $fids = $this->exporterService->collectFileIds($entity->getWebform(), $entity);
      $all_fids = array_merge($all_fids, $fids);

      // Collect signatures real paths.
      foreach ($elements as $key => $element) {
        if (isset($element['#type']) && $element['#type'] === 'webform_signature') {
          $raw = $entity->getData()[$key] ?? NULL;
          if ($raw !== NULL) {
            $path = $this->exporterService->getSignatureRealpath($entity->getWebform(), $entity, $element, $raw);
            if ($path) {
              $signature_paths[] = $path;
            }
          }
        }
      }
    }

    $all_fids = array_values(array_unique($all_fids));
    $signature_paths = array_values(array_unique($signature_paths));

    // Create Excel and zip.
    $excel_uri = $this->exporterService->createExcel($headers, $rows, 'webform_submissions_' . time());
    $zip_path = $this->exporterService->createZip($zip_basename, function ($archive) use ($all_fids, $signature_paths, $excel_uri) {
      $this->exporterService->addFilesToArchive($archive, $all_fids);
      $this->exporterService->addRealPathsToArchive($archive, $signature_paths);
      $archive->getArchive()->addFile($this->fileSystem->realpath($excel_uri), basename($excel_uri));
    });

    // Clean up Excel.
    $this->fileSystem->delete($excel_uri);

    // Stream ZIP to the browser.
    $headers = [
      'Content-Type' => 'application/zip',
      'Content-Disposition' => 'attachment; filename="' . basename($zip_path) . '"',
      'Content-Description' => 'File Transfer',
    ];
    $response = new BinaryFileResponse($zip_path, 200, $headers, TRUE);
    $response->send();
  }

  /**
   * {@inheritdoc}
   */
  public function execute($entity = NULL) {
    // This function is required but won't be used for bulk operations.
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    $result = $object->access('update', $account, TRUE);
    return $return_as_object ? $result : $result->isAllowed();
  }

}
