<?php

namespace Drupal\complete_webform_exporter\Service;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Archiver\ArchiverManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGenerator;
use Drupal\Core\Site\Settings;
use Drupal\Core\StreamWrapper\StreamWrapperManager;
use Drupal\webform\Entity\Webform;
use Drupal\webform\Entity\WebformSubmission;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

/**
 * Shared exporter logic used by controller and action plugin.
 */
class ExporterService {

  /**
   * Element types that should be skipped.
   */
  protected array $skipElementTypes = [
    'markup',
    'container',
    'fieldset',
    'webform_wizard_page',
    'processed_text',
    'webform_actions',
    'webform_flexbox',
    'hidden',
    'webform_terms_of_service',
    'webform_message',
    'value',
    'webform_variant',
    'label',
  ];

  /**
   * File elements types that should be exported as-is.
   */
  protected array $fileElementTypes = [
    'managed_file',
    'webform_document_file',
    'webform_audio_file',
    'webform_image_file',
    'webform_video_file',
    'webform_signature',
  ];

  public function __construct(
    protected readonly ArchiverManager $archiverManager,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly FileSystemInterface $fileSystem,
    protected readonly FileUrlGenerator $fileUrlGenerator,
    protected readonly StreamWrapperManager $streamWrapperManager,
  ) {}

  /**
   * Collects file IDs from all managed-file elements for a submission.
   *
   * @return int[]
   *   Array of file ids (maybe empty).
   */
  public function collectFileIds(Webform $webform, WebformSubmission $submission): array {
    $fids = [];
    $elements_managed_files = $webform->getElementsManagedFiles();
    foreach ($elements_managed_files as $element_name) {
      $file_ids = $submission->getElementData($element_name);
      if (!is_array($file_ids)) {
        if (!empty($file_ids) && is_numeric($file_ids)) {
          $file_ids = [$file_ids];
        }
        else {
          $file_ids = [];
        }
      }
      $fids = array_merge($fids, $file_ids);
    }
    // Filter non-numeric and unique.
    $fids = array_values(array_unique(array_filter($fids, 'is_numeric')));
    return $fids;
  }

  /**
   * Build headers from elements.
   *
   * @return string[]
   *   Returns an array of headers
   */
  public function buildHeaders(array $elements): array {
    $headers = [
      'Serial number',
      'Submission ID',
      'Created',
      'User',
      'LANGUAGE',
      'IP ADDRESS',
    ];

    foreach ($elements as $element) {
      if (!isset($element['#type']) || in_array($element['#type'], $this->skipElementTypes, TRUE)) {
        continue;
      }
      $headers[] = $element['#admin_title'] ?? $element['#title'] ?? $element['#key'] ?? '';
    }

    return $headers;
  }

  /**
   * Build a row data array for a single submission.
   *
   * Values for signature elements are absolute URLs. Non-file and non-signature
   * fields are output as strings (arrays imploded by ', ').
   *
   * @return string[]
   *   Row data.
   */
  public function buildRow(Webform $webform, WebformSubmission $submission, array $elements): array {
    $data = $submission->getData();
    $row = [];

    // Fixed columns.
    $row[] = $submission->get('serial')->value;
    $row[] = $submission->id();
    $row[] = date('d-m-Y H:i:s', $submission->get('created')->value);
    $row[] = $submission->getOwner() ? $submission->getOwner()->getDisplayName() : '';
    $row[] = $submission->language()->getId();
    $row[] = $submission->get('remote_addr')->value;

    // Dynamic columns.
    foreach ($elements as $key => $element) {
      if (!isset($element['#type']) || in_array($element['#type'], $this->skipElementTypes, TRUE)) {
        continue;
      }

      // Default empty.
      $value = '';

      // If the submission doesn't have the key, leave blank.
      if (!array_key_exists($key, $data)) {
        $row[] = $value;
        continue;
      }

      $raw = $data[$key];

      // Normalize arrays to comma separated strings for non-file elements.
      if (!in_array($element['#type'], $this->fileElementTypes, TRUE)) {
        $value = is_array($raw) ? implode(', ', $raw) : $raw;

        if (in_array($element['#type'], ['radios', 'select'], TRUE)) {
          $options = isset($element['#options']) && is_array($element['#options']) ? $element['#options'] : [];
          if ($value !== '' && array_key_exists($value, $options)) {
            $value = $options[$value];
          }
        }

        $row[] = $value;
        continue;
      }

      // File elements handling.
      if ($element['#type'] === 'webform_signature') {
        $signature_token = is_array($raw) ? (string) reset($raw) : (string) $raw;
        $uri_scheme = $element['#uri_scheme'] ?? 'public';
        $image_directory = "{$uri_scheme}://webform/{$webform->id()}/{$key}/{$submission->id()}";
        $image_hash = Crypt::hmacBase64('webform-signature-' . $signature_token, Settings::getHashSalt());
        $uri = "{$image_directory}/signature-{$image_hash}.png";
        $value = $this->fileUrlGenerator->generateAbsoluteString($uri);
        // We want to ensure the signature file path can be added to ZIP later.
        // We'll not add here; caller can call getSignatureRealPath() if needed.
        $row[] = $value;
        continue;
      }

      // Other managed files.
      $file_ids = !is_array($raw) ? [$raw] : $raw;
      $urls = [];
      foreach ($file_ids as $fid) {
        if (!is_numeric($fid)) {
          continue;
        }
        $file = $this->entityTypeManager->getStorage('file')->load($fid);
        if ($file) {
          $urls[] = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
        }
      }
      $row[] = implode(', ', $urls);
    }

    return $row;
  }

  /**
   * Generate signature realpath (if exists) from the signature element token.
   *
   * Returns the realpath or NULL if not available.
   */
  public function getSignatureRealpath(Webform $webform, WebformSubmission $submission, array $element, $raw_value): ?string {
    $signature_token = is_array($raw_value) ? (string) reset($raw_value) : (string) $raw_value;
    if ($signature_token === '') {
      return NULL;
    }
    $uri_scheme = $element['#uri_scheme'] ?? 'public';
    $image_directory = "{$uri_scheme}://webform/{$webform->id()}/{$element['#key']}/{$submission->id()}";
    $image_hash = Crypt::hmacBase64('webform-signature-' . $signature_token, Settings::getHashSalt());
    $uri = "{$image_directory}/signature-{$image_hash}.png";
    try {
      return $this->streamWrapperManager->getViaUri($uri)->realpath();
    }
    catch (\Throwable $e) {
      return NULL;
    }
  }

  /**
   * Create an Excel file from headers and rows.
   */
  public function createExcel(array $headers, array $rows, string $base_filename = 'webform_export'): string {
    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();

    // Write headers starting from column A.
    $colIndex = 1;
    foreach ($headers as $header) {
      $cell = Coordinate::stringFromColumnIndex($colIndex) . '1';
      $sheet->setCellValue($cell, $header);
      $colIndex++;
    }

    // Write rows starting from row 2.
    $rowIndex = 2;
    foreach ($rows as $row) {
      $colIndex = 1;
      foreach ($row as $cellValue) {
        $cell = Coordinate::stringFromColumnIndex($colIndex) . $rowIndex;
        $sheet->setCellValue($cell, $cellValue);
        $colIndex++;
      }
      $rowIndex++;
    }

    $filename = $base_filename . '-' . time() . '.xlsx';
    $uri = 'temporary://' . $filename;
    $writer = new Xlsx($spreadsheet);
    $writer->save($uri);

    return $uri;
  }

  /**
   * Add a list of file entities (by FID) to a zip archive.
   */
  public function addFilesToArchive($archive, array $fids = [], string $prefix = ''): void {
    $zip = $archive->getArchive();
    if (empty($fids)) {
      return;
    }
    $files = $this->entityTypeManager->getStorage('file')->loadMultiple($fids);
    foreach ($files as $file) {
      $real = $this->fileSystem->realpath($file->getFileUri());
      if (!empty($real) && is_file($real)) {
        $zip->addFile($real, $prefix . $file->getFilename());
      }
    }
  }

  /**
   * Add real file paths to the archive.
   */
  public function addRealPathsToArchive($archive, array $paths = [], string $prefix = ''): void {
    $zip = $archive->getArchive();
    foreach ($paths as $path) {
      if ($path && is_file($path)) {
        $zip->addFile($path, $prefix . basename($path));
      }
    }
  }

  /**
   * Create a zip file.
   */
  public function createZip(string $zip_basename, callable $addCallback): string {
    $zip_name = $zip_basename . '.zip';
    $zip_uri = 'temporary://' . $zip_name;
    $archive_path = $this->fileSystem->saveData('', $zip_uri);

    $archive_options = [
      'filepath' => $archive_path,
      'flags' => \ZipArchive::OVERWRITE,
    ];
    $archive = $this->archiverManager->getInstance($archive_options);

    // Let the caller add files via the provided callback with $archive param.
    $addCallback($archive);

    // Close archive.
    $archive->getArchive()->close();

    return $archive_path;
  }

}
