<?php

namespace Drupal\component_builder_toolbar\Routing;

use Drupal\component_builder\ComponentBuilderHelperInterface;
use Drupal\component_builder_toolbar\Form\ComponentWrapperPropertiesForm;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Routing\Route;
use Drupal\component_builder_toolbar\Form\ComponentWrapperDeleteItem;

/**
 * Provides the toolbar route.
 */
class ToolbarRoutes implements ContainerInjectionInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The component builder helper service.
   *
   * @var \Drupal\component_builder\ComponentBuilderHelperInterface
   */
  protected $componentBuilderHelper;

  /**
   * Constructs a new ToolbarRoutes object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\component_builder\ComponentBuilderHelperInterface $component_builder_helper
   *   The component builder helper service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, ComponentBuilderHelperInterface $component_builder_helper) {
    $this->entityTypeManager = $entity_type_manager;
    $this->componentBuilderHelper = $component_builder_helper;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('component_builder.helper')
    );
  }

  /**
   * Returns an array of route objects.
   *
   * @return \Symfony\Component\Routing\Route[]
   *   An array of route objects.
   */
  public function routes() {
    $routes = [];

    $referenced_entities = $this->componentBuilderHelper->getFieldsReferencedComponent();
    foreach ($referenced_entities as $entity_type_id => $entity_info) {
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      if ($detail_link = $entity_type->getLinkTemplate('canonical')) {
        $routes["entity.{$entity_type_id}.component_builder_toolbar"] = new Route(
          $detail_link . '/component-builder-toolbar',
          [
            '_controller' => '\Drupal\component_builder_toolbar\Controller\ToolBarController::page',
            '_title' => 'Component Builder Toolbar',
            'entity_type_id' => $entity_type_id,
          ],
          [
            '_entity_access' => "{$entity_type_id}.update",
          ],
          [
            'parameters' => [
              $entity_type_id => ['type' => 'entity:' . $entity_type_id],
            ],
          ]
        );
        $routes["entity.{$entity_type_id}.component_builder_toolbar_move_component"] = new Route(
          $detail_link . '/component-builder-toolbar/move/wrapper/{component_wrapper}/{field_name}/{from_region}/{from}/{to_region}/{to}',
          [
            '_controller' => '\Drupal\component_builder_toolbar\Controller\ToolBarController::moveWrapper',
            '_title' => 'Move wrapper',
            'entity_type_id' => $entity_type_id,
            'field_name' => NULL,
            'from_region' => NULL,
            'from' => NULL,
            'to_region' => NULL,
            'to' => NULL,
          ],
          [
            '_entity_access' => "{$entity_type_id}.update",
          ],
          [
            'parameters' => [
              $entity_type_id => ['type' => 'entity:' . $entity_type_id],
              'component_wrapper' => ['type' => 'entity:component_wrapper'],
            ],
          ]
        );
        $routes["entity.{$entity_type_id}.component_builder_toolbar_add_component"] = new Route(
          $detail_link . '/component-builder-toolbar/add/wrapper/{field_name}',
          [
            '_controller' => '\Drupal\component_builder_toolbar\Controller\ToolBarController::addNewComponent',
            '_title' => 'Add new component',
            'entity_type_id' => $entity_type_id,
            'field_name' => NULL,
          ],
          [
            '_entity_access' => "{$entity_type_id}.update",
          ],
          [
            'parameters' => [
              $entity_type_id => ['type' => 'entity:' . $entity_type_id],
            ],
          ]
        );
        $routes["entity.{$entity_type_id}.component_builder_toolbar_edit_component"] = new Route(
          $detail_link . '/component-builder-toolbar/edit/wrapper/{field_name}/{component_wrapper}',
          [
            '_controller' => '\Drupal\component_builder_toolbar\Controller\ToolBarController::editComponent',
            '_title' => 'Edit component',
            'entity_type_id' => $entity_type_id,
            'component_wrapper' => NULL,
            'field_name' => NULL,
          ],
          [
            '_entity_access' => "{$entity_type_id}.update",
          ],
          [
            'parameters' => [
              $entity_type_id => ['type' => 'entity:' . $entity_type_id],
              'component_wrapper' => ['type' => 'entity:component_wrapper'],
            ],
          ]
        );
        $routes["entity.{$entity_type_id}.component_builder_toolbar_delete_component_field"] = new Route(
          $detail_link . '/component-builder-toolbar/delete/wrapper/{component_wrapper}/{field_name}/{delta}',
          [
            '_form' => ComponentWrapperDeleteItem::class,
            '_title' => 'Delete wrapper item',
            'entity_type_id' => $entity_type_id,
            'field_name' => NULL,
            'delta' => NULL,
          ],
          [
            '_entity_access' => "{$entity_type_id}.update",
          ],
          [
            'parameters' => [
              $entity_type_id => ['type' => 'entity:' . $entity_type_id],
              'component_wrapper' => ['type' => 'entity:component_wrapper'],
            ],
          ]
        );
        $routes["entity.{$entity_type_id}.component_builder_toolbar_edit_properties"] = new Route(
          $detail_link . '/component-builder-toolbar/edit/wrapper/{component_wrapper}/properties/{field_name}/{delta}',
          [
            '_form' => ComponentWrapperPropertiesForm::class,
            '_title' => "Edit properties" ,
            'entity_type_id' => $entity_type_id,
            'field_name' => NULL,
            'delta' => NULL,
          ],
          [
            '_permission' => "administer site configuration",
          ],
          [
            'parameters' => [
              $entity_type_id => ['type' => 'entity:' . $entity_type_id],
              'component_wrapper' => ['type' => 'entity:component_wrapper'],
            ],
          ]
        );
      }
    }

    return $routes;
  }

}
