<?php

namespace Drupal\component_fields\Plugin\Field\FieldWidget;

use Drupal\component_fields\ComponentFieldsCompilerPluginManager;
use Drupal\component_fields\Form\ComponentFieldsBaseConfigForm;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Widget for overriding fields in component fields configuration.
 *
 * @FieldWidget(
 *   id = "component_fields_override_widget",
 *   label = @Translation("Component fields override"),
 *   field_types = {
 *     "string_long"
 *   }
 * )
 */
class ComponentFieldsWidget extends WidgetBase {

  /**
   * Entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  private EntityFieldManagerInterface $entityFieldManager;

  /**
   * Compilers that do not support multi-value.
   *
   * @var array
   */
  private array $compilersWithoutMultivalue;

  /**
   * Compilers that include multi-value options.
   *
   * @var array
   */
  private array $compilersIncludingMultivalue;

  /**
   * Default compilers for each bundle.
   *
   * @var array
   */
  private array $defaultCompilersForBundle = [];

  /**
   * ComponentFieldsWidget constructor.
   *
   * @param string $plugin_id
   *   The plugin ID.
   * @param array $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Third-party settings.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager service.
   * @param \Drupal\component_fields\ComponentFieldsCompilerPluginManager $componentFieldsPluginManager
   *   The component fields compiler plugin manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    EntityFieldManagerInterface $entity_field_manager,
    ComponentFieldsCompilerPluginManager $componentFieldsPluginManager,
    ConfigFactoryInterface $config_factory,
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->entityFieldManager = $entity_field_manager;

    $available_compiler_definitions = $componentFieldsPluginManager->getDefinitions();
    foreach ($available_compiler_definitions as $definition) {
      if (!$definition["multivalue_only"]) {
        $this->compilersWithoutMultivalue[$definition['id']] = $definition['label'];
      }
      $this->compilersIncludingMultivalue[$definition['id']] = $definition['label'];
    }

    $entity_type_id = $field_definition->getTargetEntityTypeId();
    $bundle = $field_definition->getTargetBundle();
    $savedFieldsConfig = $config_factory->get(ComponentFieldsBaseConfigForm::SETTINGS)
      ?->get('fields_config') ?? [];
    if (empty($savedFieldsConfig[$entity_type_id]) || empty($savedFieldsConfig[$entity_type_id][$bundle])) {
      return;
    }
    foreach ($savedFieldsConfig[$entity_type_id][$bundle] as $item) {
      $this->defaultCompilersForBundle[$item["final"]] = $item["default_compiler"];
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('entity_field.manager'),
      $container->get('plugin.manager.component_fields_compiler'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $entity = $items->getEntity();
    $entity_type_id = $entity->getEntityTypeId();
    $bundle = $entity->bundle();
    $overrides = $items[$delta]->value ?? NULL;
    $checkboxes = $this->getBundleFormElements($entity_type_id, $bundle, $overrides);
    $element['value'] = $element + $checkboxes;

    return $element;
  }

  /**
   * Builds form elements for the fields in the specified bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle
   *   The bundle ID.
   * @param string|null $overrides
   *   The JSON encoded overrides, if any.
   *
   * @return array
   *   An array of form elements for the bundle fields.
   */
  private function getBundleFormElements(string $entity_type_id, string $bundle, ?string $overrides): array {
    $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);
    $fields_options = [];
    foreach (array_keys($this->defaultCompilersForBundle) as $field_id) {
      $fields_options[$field_id] = $field_definitions[$field_id]->label();
    }

    if ($overrides) {
      $overrides = json_decode($overrides, TRUE);
    }

    $checkboxes = [];
    foreach ($fields_options as $field_id => $field_label) {
      $field_definition = $field_definitions[$field_id];
      $cardinality = $field_definition->getFieldStorageDefinition()->getCardinality();

      if (!empty($this->defaultCompilersForBundle[$field_id])) {
        $default_for_field = $this->defaultCompilersForBundle[$field_id];
        $default_for_field_label = $this->compilersIncludingMultivalue[$default_for_field];
        $description = $this->t("Default mode: @label", ['@label' => $default_for_field_label]);
      }

      $checkboxes[$field_id] = [
        '#title' => "$field_label<br><em>($field_id)</em>",
        '#description' => $description ?? '',
        '#type' => 'radios',
        '#options' => $this->generateOptions($cardinality),
        '#default_value' => ($overrides && !empty($overrides[$field_id])) ? $overrides[$field_id] : '',
      ];
    }

    return $checkboxes;
  }

  /**
   * Generates options based on the field cardinality.
   *
   * @param int $cardinality
   *   The cardinality of the field.
   *
   * @return array
   *   An array of options for the field widget.
   */
  private function generateOptions(int $cardinality): array {
    if ($cardinality === 1) {
      $options = $this->compilersWithoutMultivalue;
    }
    else {
      $options = $this->compilersIncludingMultivalue;
    }

    $options['default'] = "Use default";

    return $options;
  }

  /**
   * Processes and sanitizes form values before saving.
   *
   * @param array $values
   *   The values submitted in the form.
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return string|null
   *   A JSON encoded string of filtered values, or NULL if no values are valid.
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    $filtered = array_filter($values[0]["value"], function ($selected_value, $field_name) {
      $is_empty = empty($selected_value);
      $is_explicitly_default = ($selected_value === 'default');
      $is_implicitly_default = (!empty($this->defaultCompilersForBundle[$field_name]) && $selected_value === $this->defaultCompilersForBundle[$field_name]);
      return !$is_empty && !$is_explicitly_default && !$is_implicitly_default;
    }, ARRAY_FILTER_USE_BOTH);

    return empty($filtered) ? NULL : json_encode($filtered);
  }

}
