<?php

namespace Drupal\component_fields;

use Drupal\Component\Plugin\PluginManagerInterface;
use Drupal\component_fields\DTO\ComposableField;
use Drupal\component_fields\Form\ComponentFieldsBaseConfigForm;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Service for compiling fields based on composable configurations.
 */
class CompileService implements CompileServiceInterface {

  /**
   * The entity field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  private EntityFieldManagerInterface $entityFieldManager;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  private ConfigFactoryInterface $configFactory;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  private EntityTypeBundleInfoInterface $entityTypeBundleInfo;

  /**
   * The plugin manager for component fields compilers.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  private PluginManagerInterface $manager;

  /**
   * Cached array of all composable fields.
   *
   * @var array
   */
  private array $allComposableFields = [];

  /**
   * Constructs a CompileService instance.
   *
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle info service.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $component_fields_compiler_plugin_manager
   *   The plugin manager for component fields compilers.
   */
  public function __construct(EntityFieldManagerInterface $entity_field_manager, EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory, EntityTypeBundleInfoInterface $entity_type_bundle_info, PluginManagerInterface $component_fields_compiler_plugin_manager) {
    $this->entityFieldManager = $entity_field_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->configFactory = $config_factory;
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
    $this->manager = $component_fields_compiler_plugin_manager;
  }

  /**
   * Returns all composable fields grouped by field type.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   *
   * @return array
   *   An array of grouped composable fields.
   */
  public function getAllComposableFieldsGroupped(string $entity_type_id, string $bundle_id): array {
    $composable_fields = $this->getAllComposableFields();
    $groupped_fields = [];
    foreach ($composable_fields[$entity_type_id]["bundles"][$bundle_id]["fields"] as $field_id => $field) {
      $field_type = $field['type'];
      $groupped_fields[$field_type][$field_id] = $field['label'];
    }
    return $groupped_fields;
  }

  /**
   * Returns all composable fields configured in the system.
   *
   * @return array
   *   An array of all composable fields.
   */
  public function getAllComposableFields(): array {
    if (!empty($this->allComposableFields)) {
      return $this->allComposableFields;
    }
    $fieldable_entity_types = $this->entityTypeManager->getDefinitions();
    foreach ($fieldable_entity_types as $entity_type_id => $entity_type) {
      if (!$entity_type->entityClassImplements(ContentEntityInterface::class)) {
        continue;
      }
      $this->allComposableFields[$entity_type_id] = [
        'label' => $entity_type->getLabel(),
        'bundles' => [],
      ];
      $bundles_info = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id);
      foreach ($bundles_info as $bundle_id => $bundle_info) {
        $this->allComposableFields[$entity_type_id]['bundles'][$bundle_id] = [
          'label' => $bundle_info['label'],
          'fields' => [],
        ];
        $fields = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id);
        $temp_count = [];
        foreach ($fields as $field_id => $field) {
          // Skip the base fields.
          if ($field->getFieldStorageDefinition()->isBaseField()) {
            continue;
          }
          $field_type = $field->getType();
          if (!isset($temp_count[$field_type])) {
            $temp_count[$field_type] = 0;
          }
          $temp_count[$field_type]++;
          $this->allComposableFields[$entity_type_id]['bundles'][$bundle_id]['fields'][$field_id] = [
            'label' => $field->getLabel(),
            'type' => $field_type,
          ];
        }
        // If there are no fields, or if the bundle has only one field, remove
        // it from the array.
        if (empty($this->allComposableFields[$entity_type_id]['bundles'][$bundle_id]['fields']) || count($this->allComposableFields[$entity_type_id]['bundles'][$bundle_id]['fields']) === 1) {
          unset($this->allComposableFields[$entity_type_id]['bundles'][$bundle_id]);
          continue;
        }
        // The $temp_count array must have at least one element with value > 2.
        // If not, remove the bundle from the array.
        if (count(array_filter($temp_count, fn($value) => $value > 2)) === 0) {
          unset($this->allComposableFields[$entity_type_id]['bundles'][$bundle_id]);
        }
        if (isset($this->allComposableFields[$entity_type_id]['bundles'][$bundle_id])) {
          // Remove from the "fields", all the fields that are not repeated.
          $this->allComposableFields[$entity_type_id]['bundles'][$bundle_id]['fields'] = array_filter($this->allComposableFields[$entity_type_id]['bundles'][$bundle_id]['fields'], fn($field) => $temp_count[$field['type']] > 2);
        }
      }
      // If there are no bundles, remove the entity type from the array.
      if (empty($this->allComposableFields[$entity_type_id]['bundles'])) {
        unset($this->allComposableFields[$entity_type_id]);
      }
    }
    return $this->allComposableFields;
  }

  /**
   * Returns string_long fields for a given entity type and bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   *
   * @return array
   *   An array of string_long fields.
   */
  public function getStringLongFields(string $entity_type_id, string $bundle_id): array {
    $long_text_fields = [];
    $all_fields = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id);
    foreach ($all_fields as $field_id => $field) {
      // Skip the base fields.
      if ($field->getFieldStorageDefinition()->isBaseField()) {
        continue;
      }
      $field_type = $field->getType();
      if ($field_type !== 'string_long') {
        continue;
      }
      $field_label = $field->getLabel();
      $long_text_fields[$field_id] = $field_label;
    }
    return $long_text_fields;
  }

  /**
   * Returns the configured composable fields for a given entity type, bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   *
   * @return \Drupal\component_fields\DTO\ComposableField[]
   *   An array of configured composable fields.
   */
  private function getConfiguredComposableFields(string $entity_type_id, string $bundle_id): array {
    // @todo the returning array must be validated: All the component names must be checked
    //   that fields actually exist.
    $saved_config = $this->configFactory->get(ComponentFieldsBaseConfigForm::SETTINGS)
      ?->get('fields_config') ?? [];
    if (empty($saved_config)) {
      return [];
    }
    if (empty($saved_config[$entity_type_id])) {
      return [];
    }
    if (empty($saved_config[$entity_type_id][$bundle_id])) {
      return [];
    }
    $fields = [];
    foreach ($saved_config[$entity_type_id][$bundle_id] as $triad) {
      $fields[$triad["final"]] = new ComposableField($triad["component_1"], $triad["component_2"], $triad["final"], $triad["default_compiler"]);
    }
    return $fields;
  }

  /**
   * Returns the override field for a given entity type and bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   *
   * @return string|null
   *   The override field, or NULL if not found.
   */
  private function getOverrideField(string $entity_type_id, string $bundle_id): ?string {
    $overrides = $this->configFactory->get(ComponentFieldsBaseConfigForm::SETTINGS)
      ?->get('overrides') ?? [];
    if (!empty($overrides[$entity_type_id]) && !empty($overrides[$entity_type_id][$bundle_id])) {
      return $overrides[$entity_type_id][$bundle_id];
    }
    return NULL;
  }

  /**
   * Compiles the fields for a given entity based on configur composable fields.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to compile fields for.
   */
  public function compileFields(ContentEntityInterface $entity): void {
    $local_overrides = [];
    $entity_type_id = $entity->getEntityTypeId();
    $bundle_id = $entity->bundle();

    // @todo test the overrides
    $override_field = $this->getOverrideField($entity_type_id, $bundle_id);
    if (!empty($override_field)) {
      $local_overrides = $entity->{$override_field}->value;
      if ($local_overrides) {
        $local_overrides = json_decode($local_overrides, TRUE);
      }
    }

    $configured_composable_fields = $this->getConfiguredComposableFields($entity_type_id, $bundle_id);
    foreach ($configured_composable_fields as $final_field_id => $dto) {
      $component_1_value = $entity->get($dto->component1Id)->getValue();
      $component_2_value = $entity->get($dto->component2Id)->getValue();

      $compiler_id = $dto->compiler;
      if (!empty($local_overrides) && !empty($local_overrides[$final_field_id])) {
        $compiler_id = $local_overrides[$final_field_id];
      }

      $compiler = $this->manager->createInstance($compiler_id);
      $new_final_value = $compiler->compile($component_1_value, $component_2_value, $entity->getFieldDefinition($dto->finalId), $entity);
      $entity->set($final_field_id, $new_final_value);
    }
  }

}
