<?php

namespace Drupal\component_fields\Form;

use Drupal\component_fields\CompileService;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form for configuring which bundles should be enabled for component fields.
 */
class BundlesForm extends ComponentFieldsBaseConfigForm {

  /**
   * The compile service for retrieving composable fields.
   *
   * @var \Drupal\component_fields\CompileService
   */
  protected CompileService $compileService;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'component_fields_enabled_bundles_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $enabled_bundles = $this->config(static::SETTINGS)
      ->get('enabled_bundles') ?? [];
    $all_composable_fields = $this->compileService->getAllComposableFields();

    if (empty($all_composable_fields)) {
      $form["header"] = [
        '#markup' => $this->t("To be able to enable bundles, they need to have at least 3 fields of the same type."),
      ];
      return $form;
    }

    $form["header"] = [
      '#markup' => $this->t("Select the bundles that should be made available for component fields configuration. Only bundles with at least 3 fields of the same type can be enabled."),
    ];

    foreach ($all_composable_fields as $entity_type_id => $entity_type) {
      $form[$entity_type_id] = [
        '#type' => 'details',
        '#tree' => TRUE,
        '#title' => $entity_type['label'] . " [$entity_type_id]",
        '#open' => FALSE,
      ];
      foreach ($entity_type['bundles'] as $bundle_id => $bundle) {
        $form[$entity_type_id][$bundle_id] = [
          '#type' => 'checkbox',
          '#title' => $bundle['label'] . " [$bundle_id]",
          '#default_value' => in_array($bundle_id, $enabled_bundles[$entity_type_id] ?? [], TRUE),
        ];
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config(ComponentFieldsBaseConfigForm::SETTINGS);

    $values = $form_state->getValues();
    $new_enabled_bundles = [];

    foreach ($this->compileService->getAllComposableFields() as $entity_type_id => $data) {
      $enabled_bundles = isset($values[$entity_type_id]) ? array_keys(array_filter($values[$entity_type_id])) : [];
      if (!empty($enabled_bundles)) {
        $new_enabled_bundles[$entity_type_id] = $enabled_bundles;
      }
    }

    // Check the configured fields, and remove any field whose bundle has been
    // disabled.
    $fields_config = $config->get('fields_config') ?? [];
    foreach ($fields_config as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle_id => $fields) {
        if (!isset($new_enabled_bundles[$entity_type_id]) || !in_array($bundle_id, $new_enabled_bundles[$entity_type_id], TRUE)) {
          unset($fields_config[$entity_type_id][$bundle_id]);
        }
      }
      if (empty($fields_config[$entity_type_id])) {
        unset($fields_config[$entity_type_id]);
      }
    }

    // Repeat the same for the overrides.
    $overrides = $config->get('overrides') ?? [];
    foreach ($overrides as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle_id => $field_id) {
        if (!isset($new_enabled_bundles[$entity_type_id]) || !in_array($bundle_id, $new_enabled_bundles[$entity_type_id], TRUE)) {
          unset($overrides[$entity_type_id][$bundle_id]);
        }
      }
      if (empty($overrides[$entity_type_id])) {
        unset($overrides[$entity_type_id]);
      }
    }

    $this->config(static::SETTINGS)
      ->set('enabled_bundles', $new_enabled_bundles)
      ->set('fields_config', $fields_config)
      ->set('overrides', $overrides)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
