<?php

namespace Drupal\component_fields\Form;

use Drupal\component_fields\CompileService;
use Drupal\component_fields\ComponentFieldsCompilerPluginManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Form for managing component fields configuration.
 */
class FieldsForm extends ComponentFieldsBaseConfigForm {

  /**
   * List of enabled bundles for component fields.
   *
   * @var array
   */
  protected array $enabledBundles;

  /**
   * Indicates whether this is the initial build of the form.
   *
   * @var bool
   */
  protected bool $isInitialBuild = TRUE;

  /**
   * Saved configuration for fields.
   *
   * @var array
   */
  protected array $savedFieldsConfig = [];

  /**
   * Name of the final field for triads, if applicable.
   *
   * @var string|null
   */
  private ?string $triadsFinalFieldName = NULL;

  /**
   * Constructs a FieldsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   * @param \Drupal\component_fields\CompileService $compile_service
   *   The compile service for processing fields.
   * @param \Drupal\component_fields\ComponentFieldsCompilerPluginManager $compiler_manager
   *   The compiler manager for component fields.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle info service.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, CompileService $compile_service, ComponentFieldsCompilerPluginManager $compiler_manager, EntityTypeManagerInterface $entity_type_manager, EntityTypeBundleInfoInterface $entity_type_bundle_info, EntityFieldManagerInterface $entity_field_manager) {
    parent::__construct($config_factory, $compile_service, $compiler_manager, $entity_type_manager, $entity_type_bundle_info, $entity_field_manager);
    $config = $this->config(ComponentFieldsBaseConfigForm::SETTINGS);
    $this->enabledBundles = $config->get('enabled_bundles') ?? [];
    $this->savedFieldsConfig = $config->get('fields_config') ?? [];
  }

  /**
   * Returns the form ID for the component fields configuration form.
   *
   * @return string
   *   The form ID.
   */
  public function getFormId() {
    return 'component_fields_fields_config_form';
  }

  /**
   * Builds the configuration form for component fields.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The modified form array.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    if (empty($this->enabledBundles)) {
      $form['header'] = [
        '#markup' => $this->t('No bundles enabled.'),
      ];
      return $form;
    }

    $form['#tree'] = TRUE;
    $form['header'] = [
      '#markup' => $this->t('Configure the field components for each bundle.'),
    ];

    foreach ($this->enabledBundles as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle_id) {
        $this->buildTriads($form, $form_state, $entity_type_id, $bundle_id);
      }
    }

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Submit'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Builds the triad elements for the specified entity type and bundle.
   *
   * @param array &$form
   *   The form array, passed by reference.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   */
  public function buildTriads(array &$form, FormStateInterface $form_state, string $entity_type_id, string $bundle_id): void {
    $saved_triads = $this->getSavedTriads($entity_type_id, $bundle_id);
    $saved_triads_count = count($saved_triads);
    $current_triads = $this->getCurrentTriads($form_state, $entity_type_id, $bundle_id);

    if ($current_triads === NULL) {
      // The form has not been changed.
      $this->isInitialBuild = TRUE;
      if ($saved_triads_count > 0) {
        $form_triads = array_keys($saved_triads);
      }
      else {
        // Initialize the form with one empty triad.
        $form_triads = [0 => 0];
      }
    }
    else {
      // The form has been changed.
      $this->isInitialBuild = FALSE;
      $form_triads = $current_triads;
    }

    // Add the triads to the storage.
    foreach ($form_triads as $triad_index) {
      $this->addTriad($form_state, $entity_type_id, $bundle_id, $triad_index);
    }

    // Construct the form.
    $open = FALSE;
    $trigger = $form_state->getTriggeringElement();
    $parents = $trigger['#array_parents'] ?? [NULL, NULL];
    [$last_triggering_entity_type_id, $last_triggering_bundle_id] = $parents;
    if ($last_triggering_entity_type_id === $entity_type_id && $last_triggering_bundle_id === $bundle_id) {
      $open = TRUE;
    }

    $wrapper_id = "$entity_type_id-$bundle_id-fieldset-wrapper";

    $entity_type_def = $this->entityTypeManager->getDefinition($entity_type_id);
    $entity_type_label = $entity_type_def->getLabel();
    $bundle_def = $this->entityTypeBundleInfo->getBundleInfo($entity_type_id)[$bundle_id];
    $bundle_label = $bundle_def['label'];
    $fieldset_label = "$entity_type_label: $bundle_label [$entity_type_id: $bundle_id]";

    $form[$entity_type_id][$bundle_id]['triads_fieldset'] = [
      '#type' => 'details',
      '#open' => $open,
      '#title' => $fieldset_label,
      '#prefix' => '<div id="' . $wrapper_id . '">',
      '#suffix' => '</div>',
    ];
    foreach ($form_triads as $triad_index) {
      $form[$entity_type_id][$bundle_id]['triads_fieldset']['triads'][$triad_index] = $this->getTriad($form_state, $entity_type_id, $bundle_id, $triad_index);
    }

    $form[$entity_type_id][$bundle_id]['triads_fieldset']['actions'] = [
      '#type' => 'actions',
    ];
    $form[$entity_type_id][$bundle_id]['triads_fieldset']['actions']['add'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add compilable field'),
      '#submit' => ['::addSubmitCallback'],
      '#name' => "add-$entity_type_id-$bundle_id",
      '#ajax' => [
        'callback' => '::addRemoveTriadAjaxCallback',
        'wrapper' => $wrapper_id,
      ],
    ];
  }

  /**
   * Retrieves available compiler options based on the entity type and bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   *
   * @return array
   *   An associative array of compiler options.
   */
  private function getAvailableCompilerOptions($entity_type_id, $bundle_id): array {
    $options = [];
    if ($this->triadsFinalFieldName) {
      $fields = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle_id);
      $field_definition = $fields[$this->triadsFinalFieldName];
      $cardinality = $field_definition->getFieldStorageDefinition()
        ->getCardinality();
      if ($cardinality === 1) {
        $definitions = $this->componentFieldsPluginManager->getDefinitions(exclude_multivalue: TRUE);
      }
      else {
        $definitions = $this->componentFieldsPluginManager->getDefinitions();
      }

      foreach ($definitions as $id => $definition) {
        $options[$id] = $definition['label'];
      }
    }
    return ['' => '-'] + $options;
  }

  /**
   * Retrieves available field options for a specific triad component.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param int $triad_index
   *   The index of the triad.
   * @param string $component
   *   The component name (e.g., 'final', 'component_1', 'component_2').
   *
   * @return array
   *   An associative array of field options.
   */
  private function getAvailableFieldOptions(FormStateInterface $form_state, $entity_type_id, $bundle_id, $triad_index, $component): array {
    // @todo Maybe check also for cardinality.
    // First get all the fields that are available for the current entity type
    // and bundle.
    $bundle_fields_groupped = $this->compileService->getAllComposableFieldsGroupped($entity_type_id, $bundle_id);
    $bundle_fields_flat = [];
    foreach ($bundle_fields_groupped as $fields) {
      $bundle_fields_flat = array_merge($bundle_fields_flat, $fields);
    }

    // Restrict the non-final components to only fields of same type as their
    // final component.
    $selected_values = $form_state->getValues();
    $fields_of_same_type = NULL;
    if ($component !== "final") {
      $this->triadsFinalFieldName = NULL;
      if ($this->isInitialBuild) {
        $this->triadsFinalFieldName = $this->savedFieldsConfig[$entity_type_id][$bundle_id][$triad_index]["final"] ?? NULL;
      }
      else {
        if (!empty($selected_values[$entity_type_id]) && !empty($selected_values[$entity_type_id][$bundle_id])) {
          $this->triadsFinalFieldName = $selected_values[$entity_type_id][$bundle_id]["triads_fieldset"]["triads"][$triad_index]["triad_wrapper"]["final"] ?? NULL;
        }
      }

      if ($this->triadsFinalFieldName) {
        $field_type = $this->findFieldType($this->triadsFinalFieldName, $entity_type_id, $bundle_id);
        $groups = $this->compileService->getAllComposableFieldsGroupped($entity_type_id, $bundle_id);
        if ($field_type) {
          $fields_of_same_type = array_keys($groups[$field_type]);
        }
      }
    }

    if (!empty($fields_of_same_type)) {
      $bundle_fields_flat = array_intersect_key($bundle_fields_flat, array_flip($fields_of_same_type));
    }

    // Remove fields that are already selected in other triads.
    if ($this->isInitialBuild) {
      // This is the initial build of the form, so we need to exclude any values
      // that are already saved in the config, since they will be the default
      // value on other dropdowns, but not included in the $selected_values.
      if (!empty($this->savedFieldsConfig[$entity_type_id]) && !empty($this->savedFieldsConfig[$entity_type_id][$bundle_id])) {
        foreach ($this->savedFieldsConfig[$entity_type_id][$bundle_id] as $itr_index => $itr_data) {
          foreach ($itr_data as $itr_component => $itr_field) {
            if ($triad_index === $itr_index && $component === $itr_component) {
              continue;
            }
            else {
              unset($bundle_fields_flat[$itr_field]);
            }
          }
        }
      }
    }
    else {
      if (!empty($selected_values[$entity_type_id]) && !empty($selected_values[$entity_type_id][$bundle_id]) && !empty($selected_values[$entity_type_id][$bundle_id]["triads_fieldset"]["triads"])) {
        foreach ($selected_values[$entity_type_id][$bundle_id]["triads_fieldset"]["triads"] as $i => $triad) {
          $triad_wrapper = array_intersect_key($triad["triad_wrapper"], array_flip([
            'final',
            'component_1',
            'component_2',
          ]));
          foreach ($triad_wrapper as $my_component => $value) {
            if (!$value) {
              continue;
            }
            if ($i === $triad_index && $my_component === $component) {
              continue;
            }
            unset($bundle_fields_flat[$value]);
          }
        }
      }
    }

    $override_field = $this->config(ComponentFieldsBaseConfigForm::SETTINGS)
      ->get('overrides') ?? [];
    if (!empty($override_field[$entity_type_id]) && !empty($override_field[$entity_type_id][$bundle_id])) {
      $override_field = $override_field[$entity_type_id][$bundle_id];
      unset($bundle_fields_flat[$override_field]);
    }

    return ['' => '-'] + $bundle_fields_flat;
  }

  /**
   * Finds the field type for a given field name.
   *
   * @param string $field_name
   *   The name of the field.
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   *
   * @return string|null
   *   The field type if found, NULL otherwise.
   */
  private function findFieldType($field_name, $entity_type_id, $bundle_id): ?string {
    $groups = $this->compileService->getAllComposableFieldsGroupped($entity_type_id, $bundle_id);
    foreach ($groups as $group_field_type => $fields_in_group) {
      if (array_key_exists($field_name, $fields_in_group)) {
        return $group_field_type;
      }
    }
    return NULL;
  }

  /**
   * Builds a triad fieldset for a specific entity type and bundle.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param int $triad_index
   *   The index of the triad.
   *
   * @return array
   *   A render array representing the triad fieldset.
   */
  private function getTriad(FormStateInterface $form_state, $entity_type_id, $bundle_id, $triad_index): array {
    $parent_wrapper_id = "$entity_type_id-$bundle_id-fieldset-wrapper";

    $triad = [];
    $triad['triad_wrapper'] = [
      '#type' => 'fieldset',
      '#prefix' => "<div id=\"$entity_type_id-$bundle_id-fieldset-wrapper-$triad_index\">",
      '#suffix' => '</div>',
    ];
    $triad['triad_wrapper']['final'] = [
      '#type' => 'select',
      '#title' => $this->t('Compilable field'),
      '#options' => $this->getAvailableFieldOptions($form_state, $entity_type_id, $bundle_id, $triad_index, 'final'),
      '#default_value' => $this->savedFieldsConfig[$entity_type_id][$bundle_id][$triad_index]["final"] ?? '',
      '#ajax' => [
        'callback' => '::selectFieldAjaxCallback',
        'wrapper' => $parent_wrapper_id,
      ],
    ];
    $triad['triad_wrapper']['component_1'] = [
      '#type' => 'select',
      '#title' => $this->t('Component field 1'),
      '#options' => $this->getAvailableFieldOptions($form_state, $entity_type_id, $bundle_id, $triad_index, 'component_1'),
      '#default_value' => $this->savedFieldsConfig[$entity_type_id][$bundle_id][$triad_index]["component_1"] ?? '',
      '#ajax' => [
        'callback' => '::selectFieldAjaxCallback',
        'wrapper' => $parent_wrapper_id,
      ],
      '#states' => [
        'visible' => [
          ':input[name="' . $entity_type_id . '[' . $bundle_id . '][triads_fieldset][triads][' . $triad_index . '][triad_wrapper][final]"]' => ['filled' => TRUE],
        ],
      ],
    ];
    $triad['triad_wrapper']['component_2'] = [
      '#type' => 'select',
      '#title' => $this->t('Component field 2'),
      '#options' => $this->getAvailableFieldOptions($form_state, $entity_type_id, $bundle_id, $triad_index, 'component_2'),
      '#default_value' => $this->savedFieldsConfig[$entity_type_id][$bundle_id][$triad_index]["component_2"] ?? '',
      '#ajax' => [
        'callback' => '::selectFieldAjaxCallback',
        'wrapper' => $parent_wrapper_id,
      ],
      '#states' => [
        'visible' => [
          ':input[name="' . $entity_type_id . '[' . $bundle_id . '][triads_fieldset][triads][' . $triad_index . '][triad_wrapper][final]"]' => ['filled' => TRUE],
        ],
      ],
    ];

    $options = $this->getAvailableCompilerOptions($entity_type_id, $bundle_id);
    $triad['triad_wrapper']['default_compiler'] = [
      '#type' => 'select',
      '#title' => $this->t('Default compiler'),
      '#options' => $options,
      '#default_value' => $this->savedFieldsConfig[$entity_type_id][$bundle_id][$triad_index]["default_compiler"] ?? '',
      '#states' => [
        'visible' => [
          ':input[name="' . $entity_type_id . '[' . $bundle_id . '][triads_fieldset][triads][' . $triad_index . '][triad_wrapper][final]"]' => ['filled' => TRUE],
          ':input[name="' . $entity_type_id . '[' . $bundle_id . '][triads_fieldset][triads][' . $triad_index . '][triad_wrapper][component_1]"]' => ['filled' => TRUE],
          ':input[name="' . $entity_type_id . '[' . $bundle_id . '][triads_fieldset][triads][' . $triad_index . '][triad_wrapper][component_2]"]' => ['filled' => TRUE],
        ],
      ],
    ];

    $triad['triad_wrapper']['remove'] = [
      '#type' => 'submit',
      '#value' => $this->t('Remove'),
      '#submit' => ['::removeSubmitCallback'],
      '#name' => "remove-$entity_type_id-$bundle_id-$triad_index",
      '#ajax' => [
        'callback' => '::addRemoveTriadAjaxCallback',
        'wrapper' => $parent_wrapper_id,
      ],
    ];

    return $triad;
  }

  /**
   * Callback for adding a new triad to the form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function addSubmitCallback(array &$form, FormStateInterface $form_state): void {
    $trigger = $form_state->getTriggeringElement();
    $parents = $trigger['#array_parents'];
    [$entity_type_id, $bundle_id] = $parents;
    $current_triads = $this->getCurrentTriads($form_state, $entity_type_id, $bundle_id);
    $triad_index = empty($current_triads) ? 0 : max($current_triads) + 1;
    $this->addTriad($form_state, $entity_type_id, $bundle_id, $triad_index);
    $form_state->setRebuild();
  }

  /**
   * Callback for removing a triad from the form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function removeSubmitCallback(array &$form, FormStateInterface $form_state): void {
    $trigger = $form_state->getTriggeringElement();
    $parents = $trigger['#array_parents'];
    [$entity_type_id, $bundle_id, , , $triad_index] = $parents;
    $this->removeTriad($form_state, $entity_type_id, $bundle_id, $triad_index);
    $form_state->setRebuild();
  }

  /**
   * AJAX callback to handle the addition and removal of triads.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   A render array for the triads fieldset.
   */
  public function addRemoveTriadAjaxCallback(array &$form, FormStateInterface $form_state): array {
    $trigger = $form_state->getTriggeringElement();
    $parents = $trigger['#array_parents'];
    [$entity_type_id, $bundle_id] = $parents;
    return $form[$entity_type_id][$bundle_id]['triads_fieldset'];
  }

  /**
   * AJAX callback to update the available fields.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   A render array for the triads fieldset.
   */
  public function selectFieldAjaxCallback(array &$form, FormStateInterface $form_state): array {
    $trigger = $form_state->getTriggeringElement();
    $parents = $trigger['#array_parents'];
    [$entity_type_id, $bundle_id] = $parents;
    return $form[$entity_type_id][$bundle_id]['triads_fieldset'];
  }

  /**
   * Retrieves saved triads for a specific entity type and bundle.
   *
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   *
   * @return array
   *   An array of saved triads.
   */
  private function getSavedTriads($entity_type_id, $bundle_id): array {
    if (isset($this->savedFieldsConfig[$entity_type_id][$bundle_id])) {
      return $this->savedFieldsConfig[$entity_type_id][$bundle_id];
    }
    return [];
  }

  /**
   * Gets the current triads from the form state.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   *
   * @return array|null
   *   An array of current triads or NULL if none exist.
   */
  private function getCurrentTriads(FormStateInterface $form_state, $entity_type_id, $bundle_id): ?array {
    $storage = $form_state->getStorage();
    if (!isset($storage['current_triads'][$entity_type_id])) {
      return NULL;
    }
    if (!isset($storage['current_triads'][$entity_type_id][$bundle_id])) {
      return NULL;
    }

    return $storage['current_triads'][$entity_type_id][$bundle_id];
  }

  /**
   * Adds a new triad to the current triads in the form state.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param int $triad_index
   *   The index of the triad to add.
   */
  private function addTriad(FormStateInterface $form_state, $entity_type_id, $bundle_id, $triad_index): void {
    $storage = $form_state->getStorage();
    if (!isset($storage['current_triads'][$entity_type_id])) {
      $storage['current_triads'][$entity_type_id] = [];
    }
    if (!isset($storage['current_triads'][$entity_type_id][$bundle_id])) {
      $storage['current_triads'][$entity_type_id][$bundle_id][] = 0;
    }
    else {
      $storage['current_triads'][$entity_type_id][$bundle_id][$triad_index] = $triad_index;
    }

    $form_state->setStorage($storage);
  }

  /**
   * Removes a triad from the current triads in the form state.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param string $entity_type_id
   *   The entity type ID.
   * @param string $bundle_id
   *   The bundle ID.
   * @param int $triad_index
   *   The index of the triad to remove.
   */
  private function removeTriad(FormStateInterface $form_state, $entity_type_id, $bundle_id, $triad_index): void {
    $storage = $form_state->getStorage();
    if (!in_array($triad_index, $storage['current_triads'][$entity_type_id][$bundle_id], TRUE)) {
      return;
    }
    $key = array_search($triad_index, $storage['current_triads'][$entity_type_id][$bundle_id]);
    unset($storage['current_triads'][$entity_type_id][$bundle_id][$key]);

    // Clear the values of the removed triad from the form state.
    $values = $form_state->getValues();
    unset($values[$entity_type_id][$bundle_id]['triads_fieldset']['triads'][$triad_index]);
    $form_state->setValues($values);

    $form_state->setStorage($storage);
  }

  /**
   * Validates the form data to ensure all required fields are filled.
   *
   * This method checks that the final field, component fields,
   * are set correctly for each triad. If any of these fields are missing,
   * an error is added to the form state.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // Only run the validation if the form is getting saved.
    if ($form_state->getTriggeringElement()['#name'] !== 'op') {
      return;
    }

    parent::validateForm($form, $form_state);

    $values = $form_state->getValues();
    foreach ($this->enabledBundles as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle_id) {
        if (empty($values[$entity_type_id][$bundle_id]["triads_fieldset"])) {
          continue;
        }
        $triads = $values[$entity_type_id][$bundle_id]["triads_fieldset"]["triads"] ?? [];
        foreach ($triads as $i => $triad) {
          $final = $triad["triad_wrapper"]["final"] ?? NULL;
          $component_1 = $triad["triad_wrapper"]["component_1"] ?? NULL;
          $component_2 = $triad["triad_wrapper"]["component_2"] ?? NULL;
          $default_compiler = $triad["triad_wrapper"]["default_compiler"] ?? NULL;
          if (($final && $component_1 && $component_2 && $default_compiler) || (!$final && !$component_1 && !$component_2 && !$default_compiler)) {
            continue;
          }

          if (empty($triad["triad_wrapper"]["final"])) {
            $form_state->setError($form[$entity_type_id][$bundle_id]['triads_fieldset']['triads'][$i]['triad_wrapper']['final'], $this->t('Please set a final field.'));
          }
          if (empty($triad["triad_wrapper"]["component_1"])) {
            $form_state->setError($form[$entity_type_id][$bundle_id]['triads_fieldset']['triads'][$i]['triad_wrapper']['component_1'], $this->t('Please select a component field.'));
          }
          if (empty($triad["triad_wrapper"]["component_2"])) {
            $form_state->setError($form[$entity_type_id][$bundle_id]['triads_fieldset']['triads'][$i]['triad_wrapper']['component_2'], $this->t('Please select a component field.'));
          }
          if (empty($triad["triad_wrapper"]["default_compiler"])) {
            $form_state->setError($form[$entity_type_id][$bundle_id]['triads_fieldset']['triads'][$i]['triad_wrapper']['default_compiler'], $this->t('Please select a default compiler.'));
          }
        }
      }
    }
  }

  /**
   * Submits the form and saves the configuration for the enabled bundles.
   *
   * This method processes the form values, organizes them into a structured
   * configuration array, and removes any empty triads. It updates
   * for each entity type and bundle based on the submitted data.
   *
   * @param array $form
   *   The form array containing the form structure.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form, including the submitted values.
   *
   * @return void
   *   This method does not return a value. It modifies the configuration array
   *   for the enabled bundles directly.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = [];
    $values = $form_state->getValues();
    foreach ($this->enabledBundles as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle_id) {
        if (empty($values[$entity_type_id][$bundle_id]["triads_fieldset"])) {
          continue;
        }
        foreach ($values[$entity_type_id][$bundle_id]["triads_fieldset"]["triads"] as $i => $triad) {
          foreach ($triad["triad_wrapper"] as $component => $value) {
            if ($component === "remove") {
              continue;
            }
            $config[$entity_type_id][$bundle_id][$i][$component] = $value;
          }
          if (empty(array_filter($config[$entity_type_id][$bundle_id][$i]))) {
            unset($config[$entity_type_id][$bundle_id][$i]);
          }
        }
        if (!empty($config[$entity_type_id][$bundle_id])) {
          $config[$entity_type_id][$bundle_id] = array_values($config[$entity_type_id][$bundle_id]);
        }
        if (empty($config[$entity_type_id][$bundle_id])) {
          unset($config[$entity_type_id][$bundle_id]);
        }
      }
      if (empty($config[$entity_type_id])) {
        unset($config[$entity_type_id]);
      }
    }

    // Clean up possible override fields that make no sense anymore.
    $overrides = $this->config(ComponentFieldsBaseConfigForm::SETTINGS)
      ->get('overrides') ?? [];
    foreach ($overrides as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle_id => $field_id) {
        if (!isset($config[$entity_type_id]) || !array_key_exists($bundle_id, $config[$entity_type_id])) {
          unset($overrides[$entity_type_id][$bundle_id]);
        }
      }
      if (empty($overrides[$entity_type_id])) {
        unset($overrides[$entity_type_id]);
      }
    }

    $this->config(ComponentFieldsBaseConfigForm::SETTINGS)
      ->set('fields_config', $config)
      ->set('overrides', $overrides)
      ->save();

    parent::submitForm($form, $form_state);
  }

}

// @todo when changing any triad, check if there is content potentially affected and notify the user.
